/* v1.1
 *
 * events.c:  Events code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <stdlib.h>
#include <stdarg.h>
#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "object.h"
#include "events.h"
#include "sensors.h"
#include "smisc.h"

char *write_contact(CONTACT *);

char *write_contact(CONTACT *contact)
{
     char *arg;

     arg=AllocBuffer("write_contact");

     if (contact != NULL)
	  sprintf(arg, "%d", contact->contact_number);
     else
	  sprintf(arg, "-1");

     return arg;
}
	
void evCheckEvents(int space)
{
    XYZ pos1, pos2;
    range_t d;
    TAG *object;
    CONTACT *contact;
    struct timeval now;
    
    gettimeofday(&now, NULL);
    
    for (object=space_info[space].list; object != NULL; object=object->next) {
	
	if (Removed(object))
	    continue;
	
	if (!EventDriven(object))
	    continue;
	
	for (contact = object->contact_list; contact != NULL; 
	     contact=contact->next) {
	    
	    /*
	     * Check for objects moving inside and outside
	     * critical range
	     */
	    
	    object_position(object, &pos1, &now, 0);
	    object_position(contact->listref, &pos2, &now, 0);
	    
	    d = distance(&pos1, &pos2);
	    
	    if (contact->inside_critical && 
		d > F_INT(object->rec, critical_range)) {
		contact->inside_critical = FALSE;
		evTrigger(object, EVENT_OUTSIDE_CRITICAL, "i",
			  contact->contact_number);
		if (space_info[object->space].flags & SPACE_LOGGED)
		    log_space("%s (#%d) is out of %s (#%d)'s critical "
			      "range at range " RANGEF ".", 
			      contact->listref->rec->name, 
			      contact->listref->rec->db_obj, 
			      object->rec->name, object->rec->db_obj, 
			      d);
	    } 
	    else if (!contact->inside_critical && 
		     d <= F_INT(object->rec, critical_range)) {
		
		contact->inside_critical = TRUE;
		evTrigger(object, EVENT_INSIDE_CRITICAL, "i",
			  contact->contact_number);
		
		if (space_info[object->space].flags & SPACE_LOGGED)
		    log_space("%s (#%d) is in %s (#%d)'s critical range "
			      "at range " RANGEF ".", 
			      contact->listref->rec->name, 
			      contact->listref->rec->db_obj,
			      object->rec->name, object->rec->db_obj, d);
	    }
	    
	}
	
    }
    
    return;
}

void evListContainers(TAG *object, char *buff)
{
     char *ind = buff;
     TAG *ptr;
     CONTACT *con;

     strcpy(buff, "");

     for (ptr = space_info[object->space].list; ptr != NULL; 
	  ptr=ptr->next) {
	  for (con = ptr->contact_list; con != NULL; con=con->next) {

	       if (con->listref == object && con->inside_critical) {
		    sprintf(ind, "#%d ", ptr->rec->db_obj);
		    ind += strlen(ind);
	       }

	       /* Overflow prevention */
	       if (ind - buff >= LARGE_BUF_SIZE - 10)
		    return;
	  }
     }

     if (buff[strlen(buff) - 1] == ' ')
	  buff[strlen(buff) - 1] = '\0';

     return;
}

/*
 * See comments in message.c function msgSend for how to specify
 * the argument format.
 */
void evTrigger(TAG *object, const char *event, const char *argfmt, ...)
{
     va_list ap;
     unsigned int nargs, i;
     char **args;
     char *evbuff;
     CONTACT *con; 
     TAG *obj = NULL;
     static int limit=0;

     nargs = strlen(argfmt);	/* Determine number of arguments */
     args = (char **) pse_calloc(nargs, sizeof(char *));

     for (i = 0; i < nargs; i++) 
	  args[i] = (char *) pse_malloc(SMALL_BUF_SIZE);

     va_start(ap, argfmt);
     for (i = 0; i < nargs; i++) {
	  
	  switch (argfmt[i]) {
	  case 's':		/* String */
	       strncpy(args[i], va_arg(ap, char *), SMALL_BUF_SIZE);
	       args[i][SMALL_BUF_SIZE - 1] = '\0';
	       break;
	  case 'i':		/* Int */
	       sprintf(args[i], "%d", va_arg(ap, int));
	       break;
	  case 'f':		/* Float */
	       sprintf(args[i], "%f", va_arg(ap, double));
	       break;
	  case 'd':		/* Dbref */
	       obj = va_arg(ap, TAG *);
	       sprintf(args[i], "#%d", obj->rec->db_obj);
	       break;
          case 'c':		/* Contact */
	       con = va_arg(ap, CONTACT *);
	       if (con == NULL)
		    strcpy(args[i], "-1");
	       else
		    sprintf(args[i], "%d", con->contact_number);
	       break;

	  default:
	       /* Raise exception?  For now just don't send the message. */
	       return;
	       break;
	  }

     }
     va_end(ap);
     
     /* Increment the recursion counter */
     limit += 1;
     
     if (limit > 1) {
	 log_space("WARNING: evTrigger was called recursively (#%d:%d:%s).",
		   object->rec->db_obj, limit, event);
     }
     else {
	 /* Make sure that the server parser is ready. Only reset if
	  * not in recursion to prevent loops
	  */
	 ResetEval();
     }
     
     evbuff = pse_malloc(MAX_ATTRIBUTE_LEN);
     getEvalAttrBuf(object->rec->db_obj, event, args, nargs, evbuff);
     pse_free(evbuff);
     
     /* Decrement the recursion counter */
     limit -= 1;
     
     for (i = 0; i < nargs; i++) 
	 pse_free(args[i]);
     
     pse_free(args);
     
     return;
     
}

void evBroadcast(TAG *source, TAG *object, const char *msg,
		 const char *event)
{
     char *fargs[10];
     char *evbuff;
     TAG *ptr;
     CONTACT *sourcecontact, *objcontact;
     CONTACT *contact;
     char buff[SMALL_BUF_SIZE], buff2[SMALL_BUF_SIZE];

     for (ptr = space_info[source->space].list; ptr != NULL; 
	  ptr=ptr->next) {

	  if (ptr == source || ptr == object) continue;

	  sourcecontact = objcontact = NULL;

	  for (contact = ptr->contact_list; contact != NULL; 
	       contact=contact->next) {
	       
	       if (contact->listref == source) 
		    sourcecontact = contact;

	       if (contact->listref == object)
		    objcontact = contact;

	  }

	  if (sourcecontact == NULL && objcontact == NULL)
	       continue;
	  
	  if (Ship(ptr) && msg != NULL) {

	       if (sourcecontact != NULL) 
		    sprintf(buff, "Contact [%d]:  %s",
			    sourcecontact->contact_number, source->rec->name);

	       else
		    sprintf(buff, "An unknown source");
			
	       if (object != NULL) {

		    if (objcontact != NULL) {
			 sprintf(buff2, " %s [%d]",
				 object->rec->name, 
				 objcontact->contact_number);
		    }
		    else
			 sprintf(buff2, " an unknown target");
	       }
	       else
		    strcpy(buff2, "");

	       MFNotify(ptr->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		       "%s %s%s.", buff, msg, buff2); 

	  }	   
 
	  if (EventDriven(ptr) && event != NULL) {

	       fargs[0]=write_contact(sourcecontact);

	       if (object != NULL) 
		    fargs[1]=write_contact(objcontact);
	       else {
		    fargs[1]=AllocBuffer("evBroadcast");
		    sprintf(fargs[1], "-1");
	       }
	       
	       evbuff = pse_malloc(MAX_ATTRIBUTE_LEN);
	       getEvalAttrBuf(ptr->rec->db_obj, event, fargs, 2, evbuff);
	       pse_free(evbuff);
	       FreeBuffer(fargs[0]);
	       FreeBuffer(fargs[1]);

	  }
     }

     return;
}

void evExplosion(int space, XYZ location, int damage)
{
    XYZ pos;
    TAG *ptr;
    XYZ rel_pos;
    SPH bearing;
    float factor;
    struct timeval now;
    
    gettimeofday(&now, NULL);
    
    factor = (float) damage / 500.0;
    
    for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next) {
	
	object_position(ptr, &pos, &now, 0);
	
	if (CanSense(ptr) && distance(&location, &pos) < factor * 
	    (float) F_INT(ptr->rec, sensor_range)) {
	    
	    rel_pos.x = location.x - pos.x;
	    rel_pos.y = location.y - pos.y;
	    rel_pos.z = location.z - pos.z;
	    xyz_to_sph(rel_pos, &bearing);
	    
	    if (Ship(ptr)) {
		MFNotify(ptr->shipdata, -1, CONS_TAC, CONS_ACTIVE,
			 "An explosion was detected at "
			 "%3.1f%+3.1f, range %d.", bearing.bearing,
			 bearing.elevation, bearing.range);
	    }
	    
	    if (EventDriven(ptr))
		evTrigger(ptr, EVENT_EXPLOSION, "ii", damage, 
			  bearing.range);
	}
    }
    
    return;
}
