/* v1.1
 *
 * commands.c:  Space command interpreter.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "output.h"
#include "object.h"
#include "btree.h"
#include "commands.h"
#include "smisc.h"
#include "nav.h"
#include "shields.h"
#include "tactical.h"
#include "eng.h"
#include "damage.h"
#include "events.h"
#include "sensors.h"
#include "comm.h"
#include "flag.h"
#include "rxl/rxl.h"

#ifdef ENABLE_SHIELD_CLASSES
#include "scm/scm.h"
#endif

typedef struct command_entry {

     const char *name;
     void	(*fun)();	/* handler */
     int 	nargs;
     int 	flags;		/* type of function */

} CMD;

/* type flags */
#define FUNCTION	0x0001		
#define COMMAND		0x0002		
#define C_SHIP		0x0004		/* Caller must be a ship 
					 * (irrelevant w/o C_OBJ below)
					 */
#define C_OBJ		0x0008		/* Caller must be an active object */
#define REENTER         0x0010          /* Recursion safe */

#define SCDECL(x) static void x(TAG *object, dbref player, char **args)
#define SFDECL(x) static void x(TAG *object, void *mud, dbref player, \
				char **args)

static dbref parse_target(dbref, const char *);
dbref parse_caller(dbref);

SFDECL(funAddObject);
SFDECL(funRemoveObject);
SFDECL(funWriteObject);
SFDECL(funWriteAllObjects);
SFDECL(funSetObjectFlag);
SFDECL(funHasObjectFlag);
SFDECL(funSetShipFlag);
SFDECL(funHasShipFlag);
SCDECL(cmdSetConsoleFlags);
SFDECL(funGetConsoleFlags);
SFDECL(funHasConsoleFlag);
SFDECL(funConsoleEmit);
SFDECL(funGodSpecs);
SFDECL(funSpecs);
SFDECL(funSetSpaceName);
SFDECL(funGetSpaceName);
SFDECL(funContactList);
SFDECL(funShortContactList);
SFDECL(funContactInfoString);
SFDECL(funListContainers);
SCDECL(cmdEnableConsole);
SCDECL(cmdSetTorpAmmo);
SCDECL(cmdReportContact);
SCDECL(cmdEngToggleSafe);
SCDECL(cmdEngToggleWarn);
SCDECL(cmdEngBatteryOn);
SCDECL(cmdEngBatteryOff);
SCDECL(cmdEngStartReactor);
SCDECL(cmdEngShutdownReactor);
SCDECL(cmdEngAllocate);
SCDECL(cmdEngSetReactor);
SFDECL(funEngReactorSetting);
SFDECL(funEngAllocTac);
SFDECL(funEngAllocNav);
SFDECL(funEngAllocShields);
SFDECL(funEngAllocBatts);
SFDECL(funEngStress);
SFDECL(funEngOutput);
SFDECL(funEngBattStatus);
SFDECL(funEngBattLevel);
SFDECL(funEngBattMax);
SFDECL(funEngBattDisch);
SFDECL(funEngBattMaxDisch);
SCDECL(cmdNavDock);
SCDECL(cmdNavUndock);
SCDECL(cmdNavLand);
SCDECL(cmdNavLaunch);
SCDECL(cmdNavOrbit);
SCDECL(cmdNavSetCourse);
SCDECL(cmdNavSetWarp);

#ifdef ENABLE_SHIELD_CLASSES
#else
SCDECL(cmdShdAllocate);
SCDECL(cmdShdRaiseShield);
SCDECL(cmdShdLowerShield);
SCDECL(cmdShdCloakOn);
SCDECL(cmdShdCloakOff);
SFDECL(funShdCloakStatus);
SFDECL(funShdShieldLevel);
SFDECL(funShdShieldMaxLevel);
SFDECL(funShdShieldStatus);
SFDECL(funShdShieldAction);
SFDECL(funShdAllocShield);
SFDECL(funShdShortName);
SFDECL(funShdFullName);
SFDECL(funShdCharName);
SFDECL(funShdConfigName);
SFDECL(funShdNumber);
#endif

SFDECL(funNavOpenDoors);
SFDECL(funNavCloseDoors);
SCDECL(cmdNavSetRoll);
SFDECL(funNavRoll);
SFDECL(funNavHeading);
SFDECL(funNavSpeed);
SFDECL(funNavDoorStatus);
SFDECL(funListModules);
SCDECL(cmdWarpTable);
SCDECL(cmdTacLock);
SCDECL(cmdTacUnlock);
SCDECL(cmdTacTorpOnline);
SCDECL(cmdTacTorpOffline);
SCDECL(cmdTacGunOnline);
SCDECL(cmdTacGunOffline);
SCDECL(cmdTacFireTorp);
SCDECL(cmdTacFireGun);
SCDECL(cmdTacReloadTorp);
SCDECL(cmdTacAllocate);
SCDECL(cmdTacAutoReload);
SCDECL(cmdTacAutoCharge);

SCDECL(cmdTacTractorLock);
SCDECL(cmdTacTractorUnlock);
SCDECL(cmdTacTractorDock);
SCDECL(cmdTacTractorDraw);
SFDECL(funTacAllocTractor);
SFDECL(funTacTractorTarget);
SFDECL(funTacTractorStatus);

SFDECL(funTacAllocGuns);
SFDECL(funTacAllocTorps);
SFDECL(funTacGunQty);
SFDECL(funTacGunStatus);
SFDECL(funTacGunCharge);
SFDECL(funTacGunPower);
SFDECL(funTacTorpAmmo);
SFDECL(funTacTorpQty);
SFDECL(funTacTorpStatus);
SFDECL(funTacTorpCharge);
SFDECL(funTacTorpPower);
SFDECL(funTacLockStatus);
SFDECL(funTacReloadStatus);
SFDECL(funTacChargeStatus);
SCDECL(cmdCommSetChannel);
SCDECL(cmdCommLabelChannel);
SFDECL(funCommSend);
SFDECL(funCommFreq);
SFDECL(funCommLabel);
SCDECL(cmdDamRepair);
SCDECL(cmdDamSystemStatus);
SCDECL(cmdDamTeamStatus);
SFDECL(funScan);
SCDECL(cmdSetPosition);
SFDECL(funSensorStatus);
SFDECL(funScannerStatus);
SFDECL(funContactListNumbers);
SFDECL(funContactListDbrefs);
SFDECL(funContactInfoLevel);

SFDECL(funTransListBeamable);
SFDECL(funTransListDests);
SFDECL(funTransPermission);

SFDECL(funRange);
SFDECL(funOwnerNumber);
SFDECL(funFacingShield);

SFDECL(funFormatRangeString);
SFDECL(funMetricScale);
SFDECL(funActive);
SFDECL(funShip);
SFDECL(funWarpCost);
SFDECL(funMaxWarp);
SFDECL(funXYZtoSPH);
SFDECL(funSPHtoXYZ);
SFDECL(funName);
SFDECL(funPosition);
SFDECL(funRelPos);
SFDECL(funLookupContact);
SFDECL(funObjList);
SCDECL(cmdSpaceLock);
SFDECL(funSensorSanity);
SCDECL(cmdCauseDamage);
SCDECL(cmdExplosion);
SCDECL(cmdDrainReactor);
SCDECL(cmdCauseSystemDamage);

#ifdef ENABLE_ODOMETER
SFDECL(funNavOdometer);
SCDECL(cmdNavSetOdometer);
#endif

#ifdef ENABLE_TIME_TRACKER
SFDECL(funNavTimeActive);
SCDECL(cmdNavSetTimeActive);
#endif
SFDECL(funSpaceOptions);
SFDECL(funSetDebugChar);
SFDECL(funGetDebugChar);

SFDECL(funAddTimer);
SFDECL(funDelTimer);
SFDECL(funListTimers);

#ifdef ENABLE_TURN_BASED
SCDECL(funRunCycle);
#endif

/*
 * cmdlist:  List of space functions and commands
 */

static CMD cmdlist[] = {
/* Object/space/console commands/functions */
     {"add_object", funAddObject, 1, FUNCTION},
     {"rm_object", funRemoveObject, 1, FUNCTION},
     {"write_object", funWriteObject, 0, FUNCTION|C_OBJ|REENTER},
     {"write_all_objects", funWriteAllObjects, 1, FUNCTION|REENTER},
     {"set_objflag", funSetObjectFlag, 1, FUNCTION|C_OBJ|REENTER},
     {"set_shipflag", funSetShipFlag, 1, FUNCTION|C_OBJ|C_SHIP|REENTER},
     {"set_consoleflags", cmdSetConsoleFlags, 2, COMMAND|C_OBJ|C_SHIP},
     {"get_consoleflags", funGetConsoleFlags, 1, FUNCTION|C_OBJ|C_SHIP},
     {"objlist", funObjList, 4, FUNCTION|REENTER},
     {"space_lock", cmdSpaceLock, 2, COMMAND},
     {"set_space_name", funSetSpaceName, 2, FUNCTION|REENTER},
     {"get_space_name", funGetSpaceName, 1, FUNCTION|REENTER},
     {"enable_console",	cmdEnableConsole, 2, COMMAND|C_OBJ|C_SHIP},
     {"space_options", funSpaceOptions, 0, FUNCTION|REENTER},
     {"set_debug_char", funSetDebugChar, 1, FUNCTION|REENTER},
     {"get_debug_char", funGetDebugChar, 0, FUNCTION},

#ifdef ENABLE_TURN_BASED
     {"run_cycle", funRunCycle, 1, FUNCTION},
#endif

/* Timer functions */
     {"add_timer", funAddTimer, 4, C_OBJ|FUNCTION|REENTER},
     {"del_timer", funDelTimer, 1, C_OBJ|FUNCTION|REENTER},
     {"list_timers", funListTimers, 0, C_OBJ|FUNCTION|REENTER},

/* Communications commands/functions */
     {"console_emit", funConsoleEmit, 3, FUNCTION|C_OBJ|C_SHIP|REENTER},
     {"comm_set_channel", cmdCommSetChannel, 2, COMMAND|C_OBJ|C_SHIP},
     {"comm_label_channel", cmdCommLabelChannel, 2, COMMAND|C_OBJ|C_SHIP},
     {"comm_send", funCommSend,	2, FUNCTION|C_OBJ|C_SHIP},

     {"comm_freq", funCommFreq,	1, FUNCTION|C_OBJ|C_SHIP},
     {"comm_label", funCommLabel, 1, FUNCTION|C_OBJ|C_SHIP},

/* Damage control commands/functions */

     {"dmg_team_status", cmdDamTeamStatus, 0, COMMAND|C_OBJ|C_SHIP},
     {"dmg_system_status", cmdDamSystemStatus, 0, COMMAND|C_OBJ|C_SHIP},
     {"dmg_repair", cmdDamRepair, 2, COMMAND|C_OBJ|C_SHIP},

/* Engineering commands/functions */

     {"eng_start_reactor", cmdEngStartReactor, 1, COMMAND},
     {"eng_shutdown_reactor", cmdEngShutdownReactor, 0, COMMAND|C_OBJ|C_SHIP},
     {"eng_set_reactor", cmdEngSetReactor, 1, COMMAND|C_OBJ|C_SHIP},
     {"eng_battery_on",	cmdEngBatteryOn, 0, COMMAND|C_OBJ|C_SHIP},
     {"eng_battery_off", cmdEngBatteryOff, 0, COMMAND|C_OBJ|C_SHIP},
     {"eng_toggle_warn", cmdEngToggleWarn, 0, COMMAND|C_OBJ|C_SHIP},
     {"eng_toggle_safe", cmdEngToggleSafe, 0, COMMAND|C_OBJ|C_SHIP},
     {"eng_allocate", cmdEngAllocate, 4, COMMAND|C_OBJ|C_SHIP},

     {"eng_reactor_setting", funEngReactorSetting, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_alloc_tac", funEngAllocTac, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_alloc_nav", funEngAllocNav, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_alloc_batts", funEngAllocBatts, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_alloc_shields", funEngAllocShields,	0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_stress", funEngStress, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_output", funEngOutput, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_batt_status", funEngBattStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_batt_level",	funEngBattLevel, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_batt_max", funEngBattMax, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_batt_disch",	funEngBattDisch, 0, FUNCTION|C_OBJ|C_SHIP},
     {"eng_batt_max_disch", funEngBattMaxDisch,	0, FUNCTION|C_OBJ|C_SHIP},

/* Navigation/Shield commands/functions */

     {"nav_dock", cmdNavDock, 1, COMMAND|C_OBJ|C_SHIP},
     {"nav_undock", cmdNavUndock, 1, COMMAND},
     {"nav_land", cmdNavLand, 1, COMMAND|C_OBJ|C_SHIP},
     {"nav_launch", cmdNavLaunch, 1, COMMAND},
     {"nav_orbit", cmdNavOrbit,	1, COMMAND|C_OBJ|C_SHIP},
     {"nav_set_course",	cmdNavSetCourse, 3, COMMAND|C_OBJ|C_SHIP},
     {"nav_set_warp", cmdNavSetWarp, 1, COMMAND|C_OBJ|C_SHIP},
     {"nav_open_doors",	funNavOpenDoors, 0, FUNCTION|C_OBJ|C_SHIP},
     {"nav_close_doors", funNavCloseDoors, 0, FUNCTION|C_OBJ|C_SHIP},
#ifdef ENABLE_SHIELD_CLASSES
#else
     {"shd_allocate", cmdShdAllocate, 6, COMMAND|C_OBJ|C_SHIP},
     {"shd_raise_shield", cmdShdRaiseShield, 1,	COMMAND|C_OBJ|C_SHIP},
     {"shd_lower_shield", cmdShdLowerShield, 1,	COMMAND|C_OBJ|C_SHIP},
     {"shd_cloak_on", cmdShdCloakOn, 0,	COMMAND|C_OBJ|C_SHIP},
     {"shd_cloak_off", cmdShdCloakOff, 0, COMMAND|C_OBJ|C_SHIP},
     {"shd_alloc_shield", funShdAllocShield, 1,	FUNCTION|C_OBJ|C_SHIP},
     {"shd_cloak_status", funShdCloakStatus, 0,	FUNCTION|C_OBJ|C_SHIP},
     {"shd_shield_level", funShdShieldLevel, 1,	FUNCTION|C_OBJ|C_SHIP},
     {"shd_shield_max_level", funShdShieldMaxLevel, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_shield_status", funShdShieldStatus, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_shield_action", funShdShieldAction, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_short_name",	 funShdShortName, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_full_name", funShdFullName, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_char_name", funShdCharName, 1, FUNCTION|C_OBJ|C_SHIP},
     {"shd_config_name", funShdConfigName, 0, FUNCTION|C_OBJ|C_SHIP},
     {"shd_number", funShdNumber, 0, FUNCTION|C_OBJ|C_SHIP},
#endif
     {"nav_set_roll", cmdNavSetRoll, 1,	COMMAND|C_OBJ},
     {"nav_roll", funNavRoll, 0, FUNCTION|C_OBJ},
     {"nav_heading", funNavHeading, 0, FUNCTION|C_OBJ|C_SHIP},
     {"nav_speed", funNavSpeed,	0, FUNCTION|C_OBJ|C_SHIP},
     {"nav_door_status", funNavDoorStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"list_modules", funListModules, 1, FUNCTION},
#ifdef ENABLE_ODOMETER
     {"nav_odometer", funNavOdometer, 0, FUNCTION|C_OBJ|C_SHIP},
     {"nav_odometer_set", cmdNavSetOdometer, 1,	COMMAND|C_OBJ|C_SHIP},
#endif

#ifdef ENABLE_TIME_TRACKER
     {"nav_time_active", funNavTimeActive, 0, FUNCTION|C_OBJ|C_SHIP},
     {"nav_time_active_set", cmdNavSetTimeActive, 1, COMMAND|C_OBJ|C_SHIP},
#endif

/* Weapons commands/functions */

     {"tac_lock", cmdTacLock, 1, COMMAND|C_OBJ},
     {"tac_unlock", cmdTacUnlock, 0, COMMAND|C_OBJ},
     {"tac_torp_online", cmdTacTorpOnline, 1, COMMAND|C_OBJ|C_SHIP},
     {"tac_torp_offline", cmdTacTorpOffline, 1,	COMMAND|C_OBJ|C_SHIP},
     {"tac_gun_online",	cmdTacGunOnline, 1, COMMAND|C_OBJ|C_SHIP},
     {"tac_gun_offline", cmdTacGunOffline, 1, COMMAND|C_OBJ|C_SHIP},
     {"tac_fire_torp", cmdTacFireTorp, 5, COMMAND|C_OBJ|C_SHIP},
     {"tac_fire_gun", cmdTacFireGun, 1, COMMAND|C_OBJ|C_SHIP},
     {"tac_reload_torp", cmdTacReloadTorp, 1, COMMAND|C_OBJ|C_SHIP},
     {"tac_allocate", cmdTacAllocate, 3, COMMAND|C_OBJ|C_SHIP},
     {"tac_auto_reload", cmdTacAutoReload, 0, COMMAND|C_OBJ|C_SHIP},
     {"tac_auto_charge", cmdTacAutoCharge, 0, COMMAND|C_OBJ|C_SHIP},
     {"tac_tractor_lock", cmdTacTractorLock, 1,	COMMAND|C_OBJ|C_SHIP},
     {"tac_tractor_unlock", cmdTacTractorUnlock, 0, COMMAND|C_OBJ|C_SHIP},
     {"tac_tractor_dock", cmdTacTractorDock, 0, COMMAND|C_OBJ|C_SHIP},
     {"tac_tractor_draw", cmdTacTractorDraw, 1, COMMAND|C_OBJ|C_SHIP},

     {"tac_alloc_guns",	funTacAllocGuns, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_alloc_torps", funTacAllocTorps, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_gun_qty", funTacGunQty, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_gun_status", funTacGunStatus, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_gun_charge",	funTacGunCharge, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_gun_power", funTacGunPower, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_torp_qty", funTacTorpQty, 0,	FUNCTION|C_OBJ|C_SHIP},
     {"tac_torp_ammo", funTacTorpAmmo, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_torp_status", funTacTorpStatus, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_torp_charge", funTacTorpCharge, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_torp_power",	funTacTorpPower, 1, FUNCTION|C_OBJ|C_SHIP},
     {"tac_lock_status", funTacLockStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_reload_status", funTacReloadStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_charge_status", funTacChargeStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_alloc_tractor", funTacAllocTractor, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_tractor_target", funTacTractorTarget, 0, FUNCTION|C_OBJ|C_SHIP},
     {"tac_tractor_status", funTacTractorStatus, 0, FUNCTION|C_OBJ|C_SHIP},

/* Transporter functions */

     {"trans_list_dests", funTransListDests, 0,	FUNCTION|C_OBJ},
     {"trans_list_beamable", funTransListBeamable, 0, FUNCTION|C_OBJ},
     {"trans_permission", funTransPermission, 1, FUNCTION|C_OBJ},

/* Non console-dependent commands/functions */
     {"warp_cost", funWarpCost,	1, FUNCTION|C_OBJ|C_SHIP},
     {"max_warp", funMaxWarp, 1, FUNCTION|C_OBJ|C_SHIP},
     {"specs", funSpecs, 0, FUNCTION|REENTER|C_OBJ},
     {"godspecs", funGodSpecs, 0, FUNCTION|REENTER|C_OBJ},
     {"scan", funScan, 1, FUNCTION|C_OBJ|C_SHIP},
     {"sensor_status", funSensorStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"scanner_status",	funScannerStatus, 0, FUNCTION|C_OBJ|C_SHIP},
     {"warp_table", cmdWarpTable, 0, COMMAND|C_OBJ|C_SHIP},
     {"contact_list", funContactList, 2, FUNCTION|C_OBJ|REENTER}, 
     {"short_contact_list", funShortContactList, 2, FUNCTION|C_OBJ|REENTER}, 
     {"contact_info_string", funContactInfoString, 1, FUNCTION|C_OBJ}, 
     {"report_contact",	cmdReportContact, 1, COMMAND|C_OBJ},
     {"sensor_sanity", funSensorSanity, 1, FUNCTION},

/* General Coding functions */

     {"active",	funActive, 1, FUNCTION},
     {"ship", funShip, 0, FUNCTION|C_OBJ},
     {"contact_list_num", funContactListNumbers, 0, FUNCTION|C_OBJ},
     {"contact_list_dbrefs", funContactListDbrefs, 0, FUNCTION|C_OBJ},
     {"contact_info_level", funContactInfoLevel, 1, FUNCTION|C_OBJ},
     {"range", funRange, 1, FUNCTION|C_OBJ},
     {"owner_number", funOwnerNumber, 0, FUNCTION|C_OBJ|C_SHIP},
     {"facing_shield", funFacingShield,	1, FUNCTION|C_OBJ},
     {"pos", funPosition, 0, FUNCTION|C_OBJ},
     {"set_pos", cmdSetPosition, 3, COMMAND|C_OBJ},
     {"rel_pos", funRelPos, 1, FUNCTION|C_OBJ},
     {"lookup_contact",	funLookupContact, 1, FUNCTION|C_OBJ},
     {"has_obj_flag", funHasObjectFlag,	1, FUNCTION|C_OBJ},
     {"has_ship_flag", funHasShipFlag, 1, FUNCTION|C_OBJ|C_SHIP},
     {"has_console_flag", funHasConsoleFlag, 2, FUNCTION|C_OBJ|C_SHIP},
     {"cause_damage", cmdCauseDamage, 3, COMMAND|C_OBJ},
     {"cause_system_damage", cmdCauseSystemDamage, 2, COMMAND|C_OBJ|C_SHIP},
     {"explosion", cmdExplosion, 5, COMMAND},
     {"drain_reactor", cmdDrainReactor,	1, COMMAND|C_OBJ|C_SHIP},
     {"name", funName, 0, FUNCTION|C_OBJ},
     {"list_containers", funListContainers, 0, FUNCTION|C_OBJ},
     {"set_torp_ammo", cmdSetTorpAmmo, 1, COMMAND|C_OBJ|C_SHIP},

/* Miscellaneous functions */
     {"xyz_to_sph", funXYZtoSPH, 3, FUNCTION},
     {"sph_to_xyz", funSPHtoXYZ, 3, FUNCTION},
     {"format_range_string", funFormatRangeString, 3, FUNCTION},
     {"metric_scale", funMetricScale, 3, FUNCTION},

/* End-of-list marker */
     {"", NULL,	0}};

static BNODE *fun_htab;

/*
 * Called as part of initialising space. It builds a btree table for quick
 * lookup of spacecalls.
 */
void cmdInitFunctionTable(void)
{
    int i;
    
    fun_htab = NULL;
    for (i=0; *(cmdlist[i].name) != '\0'; i++) 
	btreeInsert(cmdlist[i].name, (void *)(cmdlist + i), &fun_htab);
    
    return;
}

/*
 * This is called by the MU* server to handle the sc() function.  MUDFUNCTION
 * is platform specific and defined in platform.h
 */
void spacecall(void *mud, dbref Player, dbref Cause, const char *Args[],
	       int NumArgs)
{
    dbref data_object;
    int offset = 0;
    TAG *object = NULL;
    CMD *fp;
    
    if (!IsSpace(Cause) && !IsSpace(Player)) {
	WriteFunctionBuffer("#-1 PERMISSION DENIED");
	return;
    }
    
    if (NumArgs < 1) {
	WriteFunctionBuffer("#-1 FUNCTION(SC) REQUIRES AT LEAST ONE "
			    "ARGUMENT");
	return;
    }
    
    fp = (CMD *)btreeSearch(Args[0], fun_htab);
    if (fp == NULL) {
	FWriteFunctionBuffer("#-1 UNKNOWN SPACE FUNCTION(%s)", Args[0]);
	return;
    }
    
    /* If the function only works on objects, find the object.
     *
     * This is slightly complicated by the fact that PSE can use either an
     * explicitly provided space object, or using the DataDBREF attribute.
     * Hence, the check for the right number of attributes depends on this
     * too. */
    if ((fp->flags & C_OBJ) != 0) {
	
	if (NumArgs == fp->nargs + 2) {
	    data_object = parse_target(Player, Args[1]);
	    offset = 1;
	} else if (NumArgs == fp->nargs + 1) {
	    data_object = parse_caller(Player);
	} else {
	    FWriteFunctionBuffer2("#-1 SPACE FUNCTION(%s) REQUIRES %d "
				  "ARGUMENTS", fp->name, fp->nargs);
	    return;
	}
	
	if (!ValidObject(data_object)) {
	    FWriteFunctionBuffer("#-1 BAD DATA OBJECT #%d", data_object);
	    return;
	}
	
	object = objFindObject(data_object);
	
	if ((object==NULL) || Removed(object)) {
	    FWriteFunctionBuffer("#-1 OBJECT #%d NOT ACTIVE", data_object);
	    return;
	}
	
	if ((fp->flags & C_SHIP) != 0 && !Ship(object)) {
	    FWriteFunctionBuffer("#-1 SPACE FUNCTION(%s) FOR SHIPS ONLY",
				 fp->name);
	    return;
	}
    }
    else if (NumArgs != fp->nargs + 1) { 
	
	FWriteFunctionBuffer2("#-1 SPACE FUNCTION(%s) REQUIRES %d ARGUMENTS",
			      fp->name, fp->nargs); 
	return;
    }
    
    /* Run the requested function. */

    if ((fp->flags & COMMAND) != 0)
	CALL_COMMAND(fp->fun, object, Cause, Args + offset);
    
    else if ((fp->flags & FUNCTION) != 0)
	CALL_FUNCTION(fp->fun, object, Cause, Args + offset);
    
    return;
}

/*
 * parse_caller:  If DBREFDATA is set on the calling object, assume that 
 *		it is a console of some sort, and use the dbref in the field
 *		as the data object.  Otherwise, use the calling object's
 *		dbref.
 */
dbref parse_caller(dbref caller)
{
     dbref data;

     data = dbrefData(caller);

     if (data == PSE_NOTHING)
	  data = caller;

     return data;
}
	
/*
 * parse_target:  A target has been provided.  If the target starts with
 *		a "C", it is a contact, and we should look for the contact
 *		number relative to the caller (see parse_caller above).
 *		If the target isn't a contact, assume it's another data object.
 *		
 */
dbref parse_target(dbref data, const char *arg)
{
     TAG *object;
     CONTACT *contact;

     if (*arg == 'c' || *arg == 'C')	{
	  if ((object = objFindObject(parse_caller(data)))==NULL)
	       return PSE_NOTHING;
		
	  if ((contact = snsFindContactByNumber(object, atoi(arg+1)))==NULL)
	       return PSE_NOTHING;

	  return contact->listref->rec->db_obj;
     }
     else
	  return (parseDbref(arg));
}

/*
 * Space god commands
 */

SFDECL(funAddObject)
{
    dbref data;

    data = parseDbref(args[1]);

    if (!ValidObject(data)) {
        FWriteFunctionBuffer("#-1 Invalid object (#%d)", data);
	return;
    }

    clsAddRecord(player, data);
}

SFDECL(funRemoveObject)
{
    dbref data;
    Class *class;
    Record *recptr;

    data = parseDbref(args[1]);

    if (!ValidObject(data)) {
        FWriteFunctionBuffer("#-1 Invalid object (#%d)", data);
	return;
    }

    class = clsClassOf(data);
    if (class == NULL) {
        FWriteFunctionBuffer("#-1 Object has not been setup properly (#%d)", data);
	log_space("RemoveObject Failure: #%d: CF_CLASS not set.", data);
	return;
    }

    recptr = clsFindRecord(class, data);
    if (recptr == NULL) {
        FWriteFunctionBuffer("#-1 Object has not been added yet (#%d)", data);
	log_space("RemoveObject Failure: #%d: Object not found.", data);
	return;
    }

    if (clsRemoveRecord(recptr)) {
        FWriteFunctionBuffer("#-1 Object cannot be removed at this time (#%d)", data);
	log_space("RemoveObject Failure: #%d: Object cannot be removed.", data);
	return;
    }
}

SFDECL(funWriteObject)
{
    objUpdateState(object, mud);
}

SFDECL(funWriteAllObjects)
{
    int space;
    TAG *tag;
    
    space = atoi(args[1]);
    
    if (space < 0 || space >= NUM_SPACES) {
	Notify(player, "Invalid space.");
	return;
    }
    
    tag = space_info[space].list;
    
    while (tag != NULL) {
	objUpdateState(tag, mud);
	tag = tag->next;
    }
}
	
SFDECL(funSetObjectFlag)
{
    
    if (args[1][0] != '\0') {
	if (args[1][0] == '!')
	    objSetObjectFlag(object, mud, 0, &args[1][1]);
	else
	    objSetObjectFlag(object, mud, 1, args[1]);
    }
    else
	WriteFunctionBuffer("#-1 Invalid (null) flag specified.");
    
    return;
}

SFDECL(funHasObjectFlag)
{
    FWriteFunctionBuffer("%d", objHasObjectFlag(object, player, args[1]));
}

SFDECL(funSetShipFlag)
{
    
    if (args[1][0] != '\0') {
	if (args[1][0] == '!')
	    objSetShipFlag(object, mud, 0, &args[1][1]);
	else
	    objSetShipFlag(object, mud, 1, args[1]);
	
    }
    else
	WriteFunctionBuffer("#-1 Invalid (null) flag specified.");
    
    return;
}

SFDECL(funHasShipFlag)
{
     FWriteFunctionBuffer("%d", objHasShipFlag(object, player, args[1]));
}

SCDECL(cmdSetConsoleFlags)
{
    dbref console;
    int   i;
    SHIP  *ship = object->shipdata;
    
    console = parseDbref(args[1]);
    
    for (i = 0; i < ship->num_consoles; i ++) {
	if (ship->consoles[i].dbref == console) {
	    if (loadSymbolicFlags(&ship->consoles[i].flags, FLIST_CONSOLE,
				  args[2]) != NULL)
		Notify(player, "Invalid console type specified.");
	}
    }
}

SFDECL(funGetConsoleFlags)
{
     dbref console;
     SHIP *ship = object->shipdata;
     int i;

     console = parseDbref(args[1]);

     for (i = 0; i < ship->num_consoles; i ++)
	  if (ship->consoles[i].dbref == console) {
	       WriteFunctionBuffer(saveSymbolicFlags(ship->consoles[i].flags,
						     FLIST_CONSOLE, 1));
	       return;
	  }

     WriteFunctionBuffer("#-1 Console not found.");
}

SFDECL(funHasConsoleFlag)
{
     dbref console;
     SHIP *ship = object->shipdata;
     int i;     
     
     console = parseDbref(args[1]);

     for (i = 0; i < ship->num_consoles; i ++) {
	 if (ship->consoles[i].dbref == console)
	     FWriteFunctionBuffer("%d", objHasConsoleFlag(object, player,
							  i, args[2]));
	     return;
     }

     WriteFunctionBuffer("#-1 Console not found.");
}

SFDECL(funConsoleEmit)
{
    unsigned int aflags, oflags;
    SHIP *ship = object->shipdata;

    oflags = atoi(args[1]);
    aflags = atoi(args[2]);
    MFNotify(ship, PSE_NOTHING, oflags, aflags, "%s", args[3]);
}

SFDECL(funGodSpecs)
{
     objDisplaySpecs(object, mud, 1);
}

SFDECL(funObjList)
{
    int space;
    
    space = atoi(args[1]);
    
    if (space < 0 || space >= NUM_SPACES) {
	FWriteFunctionBuffer("#-1 INVALID SPACE(%d).", space);
	return;
    }
    
    objList(mud, space, atoi(args[2]), atoi(args[3]), atoi(args[4]));
}

SCDECL(cmdSpaceLock)
{
     int space;
	
     space = atoi(args[1]);

     if (space < 0 || space >= NUM_SPACES) {
	  FNotify(player, "Valid spaces are between 0 and %d.",
		  NUM_SPACES - 1);
	  return;
     }

     if (atoi(args[2])) {
    
	  space_info[space].flags |= SPACE_LOCKED;
	  FNotify(player, "Space %d locked.", space);

     } else {

	  space_info[space].flags &= ~SPACE_LOCKED;
	  FNotify(player, "Space %d unlocked.", space);

     }

     return;

}

SFDECL(funContactList)
{
     snsDisplayContacts(object, mud, atoi(args[1]), atoi(args[2]));
}

SFDECL(funShortContactList)
{
     snsDisplayShortContacts(object, mud, atoi(args[1]), atoi(args[2]));
}

SFDECL(funContactInfoString)
{
     char *buff;
     CONTACT *contact;
     struct timeval now;
     
     contact = snsFindContactByNumber(object, atoi(args[1]));

     if (contact == NULL)
	 WriteFunctionBuffer("#-1 BAD CONTACT");
     else {
	 
	 gettimeofday(&now, NULL);
	 
	 buff = pse_malloc(MAX_ATTRIBUTE_LEN);
	 snsContactString(object, contact, buff, &now);
	 WriteFunctionBuffer(buff);
	 pse_free(buff);
     }
}

SCDECL(cmdReportContact)
{
     CONTACT *contact;

     contact = snsFindContactByNumber(object, atoi(args[1]));

     if (contact == NULL)
	  Notify(player, "Contact not on sensors.");
     else
	  snsDisplaySensorInfo(object, contact, player, DSD_ROUTINE);

}

SCDECL(cmdEnableConsole)
{
    dbref console;
    int   i;
    SHIP *ship = object->shipdata;
    
    console = parseDbref(args[1]);
    
    for (i = 0; i < ship->num_consoles; i ++) {
	if (ship->consoles[i].dbref == console) {

	    if (atoi(args[2]) != 0) {
		ship->consoles[i].flags |= CONS_ACTIVE;
		return;
	    } else {
		ship->consoles[i].flags &= (~CONS_ACTIVE);
		return;
	    }
	}
    }

    FNotify(player, "#-1 Console #%d for ship #%d not found.", console, ship);
}

/* 
 * Basic informative commands
 */

SFDECL(funSpecs)
{
     objDisplaySpecs(object, mud, 0);
}

/*
 * Engineering commands
 */

SCDECL(cmdEngToggleSafe)
{
     SHIP *ship = object->shipdata;

     ship->eng_safeties = !ship->eng_safeties;

     if (ship->eng_safeties)
	  Notify(player, "Safety overrides enabled.");
     else
	  Notify(player, "Safety overrides disabled.");

     return;
}

SCDECL(cmdEngToggleWarn)
{
     SHIP *ship = object->shipdata;

     ship->eng_warnings = !ship->eng_warnings;

     if (ship->eng_warnings)
	  Notify(player, "Warning messages enabled.");
     else
	  Notify(player, "Warning messages disabled.");

     return;
}

SCDECL(cmdEngBatteryOn)
{
     engBatteryOn(object, player);
}

SCDECL(cmdEngBatteryOff)
{
     engBatteryOff(object, player);
}

SCDECL(cmdEngStartReactor)
{
     dbref data;

     data = parseDbref(args[1]);

     if (objFindObject(data) == NULL)
	  engStartReactor(object, player, parseDbref(args[1]));
     else
	  Notify(player, "The engine is already running.");
}

SCDECL(cmdEngShutdownReactor)
{
     engShutdownReactor(object, player);
}

SCDECL(cmdEngAllocate)
{
     engAllocate(object, player, atoi(args[1]), atoi(args[2]), atoi(args[3]),
		 atoi(args[4]));
}

SCDECL(cmdEngSetReactor)
{
     engSetReactorOutput(object, player, atoi(args[1]));
}

SFDECL(funEngReactorSetting)
{
     FWriteFunctionBuffer("%d", object->shipdata->reactor_setting);
}

SFDECL(funEngAllocTac)
{
     FWriteFunctionBuffer("%d", object->shipdata->alloc_tac);
}

SFDECL(funEngAllocNav)
{
     FWriteFunctionBuffer("%d", object->shipdata->alloc_nav);
}

SFDECL(funEngAllocBatts)
{
     FWriteFunctionBuffer("%d", object->shipdata->alloc_batt);
}

SFDECL(funEngAllocShields)
{
     FWriteFunctionBuffer("%d", object->shipdata->alloc_shields);
}

SFDECL(funEngStress)
{
     FWriteFunctionBuffer("%d", 100 * object->shipdata->overload_points /
			  F_INT(object->shipdata->rec, max_overload_points));
}

SFDECL(funEngOutput)
{
     FWriteFunctionBuffer("%d", object->shipdata->reactor_output);
}

SFDECL(funEngBattStatus)
{

     switch (object->shipdata->battery_status) {

     case BTTY_OFFLINE:
	  WriteFunctionBuffer("OFF");
	  break;
     case BTTY_ONLINE:
	  WriteFunctionBuffer("ON");
	  break;
     case BTTY_DAMAGED:
	  WriteFunctionBuffer("DAMAGED");
	  break;
     }
}

SFDECL(funEngBattLevel)
{
     FWriteFunctionBuffer("%d", object->shipdata->battery_level);
}

SFDECL(funEngBattMax)
{
     FWriteFunctionBuffer("%d", 
			  F_INT(object->shipdata->rec, battery_capacity));
}

SFDECL(funEngBattDisch)
{
     FWriteFunctionBuffer("%d", object->shipdata->battery_discharge);
}

SFDECL(funEngBattMaxDisch)
{
     FWriteFunctionBuffer("%d", 
			  F_INT(object->shipdata->rec, battery_discharge_max));
}

/*
 * Navigation commands
 */

SCDECL(cmdNavDock)
{
     CONTACT *dock;

     dock = snsFindContactByNumber(object, atoi(args[1]));

     if (dock==NULL) {
	  Notify(player, "Invalid target.");
	  return;
     }

     navDock(object, player, dock->listref, 0);
}

SCDECL(cmdNavUndock)
{
     dbref data;

     data = parseDbref(args[1]);

     if (objFindObject(data) == NULL)
	  navUndock(object, player, parseDbref(args[1]));	
     else
	  Notify(player, "The ship isn't currently docked.");
}

SCDECL(cmdNavLand)
{
     navLand(object, player, atoi(args[1]));
}

SCDECL(cmdNavLaunch)
{
     dbref data;

     data = parseDbref(args[1]);

     if (objFindObject(data) == NULL)
	  navLaunch(object, player, parseDbref(args[1]));	
     else
	  Notify(player, "The ship is not on a planet.");
}

SCDECL(cmdNavOrbit)
{
     navOrbit(object, player, atoi(args[1]));
}

SCDECL(cmdNavSetCourse)
{
     float elevation;

     elevation = atof(args[3]);

     if (strlen(args[2]) && *(args[2])=='-')
	  elevation = -elevation;

     navSetCourse(object, player, atof(args[1]), elevation);
}

SCDECL(cmdNavSetRoll)
{
     float rr;

     rr = atof(args[1]);

     if (rr < 0.0 || rr >= 360.0) {
	  Notify(player, "Roll can only be set from 0-359.");
	  return;
     }

     if (object->roll != rr) {

	  /* Calculate the roll adjustment and direction required */
	  object->roll_adj = rr - object->roll;

	  if (object->roll_adj < -180.0)
	       object->roll_adj += 360.0;

	  if (object->roll_adj > 180.0)
	       object->roll_adj -= 360.0;

	  MFNotify(object->shipdata, player, CONS_NAV, CONS_ACTIVE,
		   "[Now rolling to %d]", (int) rr);

     } else {

	  object->roll_adj = 0.0;

	  MFNotify(object->shipdata, player, CONS_NAV, CONS_ACTIVE,
		   "[Roll complete]");

     }
}

SFDECL(funNavRoll)
{
     FWriteFunctionBuffer("%3d",(int) object->roll);
}

SCDECL(cmdNavSetWarp)
{
     navSetWarp(object, player, atof(args[1]));
}

#ifdef ENABLE_SHIELD_CLASSES
#else
SCDECL(cmdShdAllocate)
{
     int shields[NUM_SHIELDS], shield;

     for (shield = 0; shield < object->shipdata->shield_number; shield ++)
	  shields[shield] = atoi(args[shield+1]);

     for ( ; shield < NUM_SHIELDS; shield ++)
	  shields[shield] = 0;

     shdAllocate(object, player, shields);
}

SCDECL(cmdShdRaiseShield)
{
     shdRaiseShield(object, player, atoi(args[1]));
}

SCDECL(cmdShdLowerShield)
{
     shdLowerShield(object, player, atoi(args[1]));
}

SCDECL(cmdShdCloakOn)
{
     shdCloakOn(object, player);
}

SCDECL(cmdShdCloakOff)
{
     shdCloakOff(object, player);
}

SFDECL(funShdShieldLevel)
{
     int shield;

     shield = atoi(args[1]);

     if (shield >= 0 && shield < object->shipdata->shield_number) 
	  FWriteFunctionBuffer("%d", 
			       object->shipdata->shield_level[shield]);
     else
	 WriteFunctionBuffer("-1");
}

SFDECL(funShdShieldMaxLevel)
{
     int shield;

     shield = atoi(args[1]);

     if (shield >= 0 && shield < object->shipdata->shield_number) 
	  FWriteFunctionBuffer("%d",
			       object->shipdata->shield_max[shield]);
     else
	 WriteFunctionBuffer("-1");
}

SFDECL(funShdShieldStatus)
{
     int shield;

     shield = atoi(args[1]);

     if (shield >= 0 && shield < object->shipdata->shield_number)
	  switch (object->shipdata->shield_status[shield]) {
	  case SHLD_READY:
	       WriteFunctionBuffer("READY");
	       break;
	  case SHLD_UP:
	       WriteFunctionBuffer("UP");
	       break;
	  case SHLD_DAMAGED:
	       WriteFunctionBuffer("DAMAGED");
	       break;
	  }

}

SFDECL(funShdShortName)
{
     int shield;

     shield = atoi(args[1]);

     if ((shield < 0) || (shield >= object->shipdata->shield_number)) {
	  WriteFunctionBuffer("#-1 Invalid Shield ID");
	  return;
     }

     WriteFunctionBuffer(shdShortName(object, shield));
}

SFDECL(funShdFullName)
{
     int shield;

     shield = atoi(args[1]);

     if ((shield < 0) || (shield >= object->shipdata->shield_number)) {
	  WriteFunctionBuffer("#-1 Invalid Shield ID");
	  return;
     }

     WriteFunctionBuffer(shdFullName(object, shield));
}

SFDECL(funShdCharName)
{
     int shield;

     shield = atoi(args[1]);

     if ((shield < 0) || (shield >= object->shipdata->shield_number)) {
	  WriteFunctionBuffer("#-1 Invalid Shield ID");
	  return;
     }

     WriteFunctionBuffer(shdCharName(object, shield));
}

SFDECL(funShdConfigName)
{
     WriteFunctionBuffer(shdConfigName(object));
}

SFDECL(funShdNumber)
{
     FWriteFunctionBuffer("%d", object->shipdata->shield_number);
}

SFDECL(funShdShieldAction)
{
     int shield;

     shield = atoi(args[1]);

     if (shield >= 0 && shield < object->shipdata->shield_number) {

	  if (object->shipdata->shield_status[shield] !=  SHLD_DAMAGED)
	       switch (object->shipdata->shield_action[shield]) {

	       case SHLD_STABLE:
		    WriteFunctionBuffer("STABLE");
		    break;
	       case SHLD_CHARGING:
		    WriteFunctionBuffer("CHARGE");
		    break;
	       case SHLD_FALLING:
		    WriteFunctionBuffer("FALL");
		    break;
	       case SHLD_OVERLOADED:
		    WriteFunctionBuffer("OVERLOAD");
		    break;
	       }

     }

}

SFDECL(funShdAllocShield)
{
     int shield;

     shield = atoi(args[1]);

     if (shield >= 0 && shield < object->shipdata->shield_number) 
	 FWriteFunctionBuffer("%d", 
			      object->shipdata->shield_alloc[shield]);
     else
	 WriteFunctionBuffer("-1");
}

#endif

SFDECL(funNavOpenDoors)
{
     navDoorControl(object, player, mud, 1);
}

SFDECL(funNavCloseDoors)
{
     navDoorControl(object, player, mud, 0);
}

SFDECL(funPosition)
{
    XYZ pos;
    char *wbuf;
    struct timeval now;

    gettimeofday(&now, NULL);
    
    object_position(object, &pos, &now, 0);

    wbuf = pse_malloc(SMALL_BUF_SIZE);
    sprintf(wbuf, RANGEF " " RANGEF " " RANGEF, pos.x, pos.y, pos.z);
    WriteFunctionBuffer(wbuf);
    pse_free(wbuf);
}

SCDECL(cmdSetPosition)
{
    XYZ pos;
    range_t dist;
    
    pos.x = RANGEI(args[1]);
    pos.y = RANGEI(args[2]);
    pos.z = RANGEI(args[3]);
    
    /*
     * Ensure that we respect the maximum size of space.
     */
    if ((RANGEA(pos.x) > MAX_RANGE) || (RANGEA(pos.y) > MAX_RANGE) ||
	(RANGEA(pos.z) > MAX_RANGE)) {
	Notify(player, "#-1 INVALID POSITION SPECIFIED");
	return;
    }

    dist = distance(&pos, &object->dc_pos);
    
    if (dist == 0)
	return;
    
    /* Flag that we moved this time */
    object->moveID = move_turn_id;
    
#ifdef ENABLE_ODOMETER
    if (Ship(object) && (dist != 0))
	F_INT(object->shipdata->rec, odometer) += dist;
#endif
    
    object->dc_pos = pos;
    gettimeofday(&object->dc_time, NULL);
}

SFDECL(funRelPos)
{
    XYZ pos1, pos2;
    char *wbuf;
    SPH bearing;
    XYZ rel_pos;
    TAG *target;
    struct timeval now;

    if ((target=objFindObject(parseDbref(args[1])))==NULL) {
	WriteFunctionBuffer("#-1 Invalid target");
	return;
    }	
    
    gettimeofday(&now, NULL);

    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);    
    
    wbuf = pse_malloc(SMALL_BUF_SIZE);
    
    rel_pos.x = pos2.x - pos1.x;
    rel_pos.y = pos2.y - pos1.y;
    rel_pos.z = pos2.z - pos1.z;

     xyz_to_sph(rel_pos, &bearing);
     sprintf(wbuf, "%1.2f %+2.2f %.2f", bearing.bearing, bearing.elevation,
	     bearing.range * F_FLOAT(object->rec, range_factor));

     WriteFunctionBuffer(wbuf);
     
     pse_free(wbuf);
}

SFDECL(funLookupContact)
{
     CONTACT *contact;

     contact = snsFindContactByNumber(object, atoi(args[1]));

     if (contact != NULL)
	  FWriteFunctionBuffer("#%d", contact->listref->rec->db_obj);
     else
	  WriteFunctionBuffer("#-1");
}

SFDECL(funNavHeading)
{
     char *wbuf;

     wbuf = pse_malloc(SMALL_BUF_SIZE);
     sprintf(wbuf, "%1.2f %+2.2f", object->heading.bearing, 
	     object->heading.elevation);

     WriteFunctionBuffer(wbuf);
     pse_free(wbuf);
}

SFDECL(funNavSpeed)
{
     FWriteFunctionBuffer("%3.2f",object->speed);
}

#ifdef ENABLE_ODOMETER

SFDECL(funNavOdometer)
{
     if (Ship(object))
	  FWriteFunctionBuffer("%d", F_INT(object->shipdata->rec, odometer));
}

SCDECL(cmdNavSetOdometer)
{
     navSetOdometer(object, atoi(args[1]));
}

#endif

#ifdef ENABLE_TIME_TRACKER

SFDECL(funNavTimeActive)
{
     time_t t;

     if (Ship(object)) {
		
	  /* Add voyage time to object active time */
	  t = time(NULL) - object->shipdata->time_start;
	  FWriteFunctionBuffer("%d", 
			       F_INT(object->shipdata->rec, time_active) + t);

     }
}

SCDECL(cmdNavSetTimeActive)
{
     if (Ship(object)) {

	  object->shipdata->time_start = time(NULL);
	  F_INT(object->shipdata->rec, time_active) = atoi(args[1]);

     }
}

#endif

SFDECL(funShdCloakStatus)
{
     switch (object->shipdata->cloak_status) {

     case CLOAK_NONE:
	  WriteFunctionBuffer("NONE");
	  break;
     case CLOAK_OFF:
	  WriteFunctionBuffer("OFF");
	  break;
     case CLOAK_ON:
	  WriteFunctionBuffer("ON");
	  break;
     case CLOAKING:
	  WriteFunctionBuffer("CLOAKING");
	  break;
     case DECLOAKING:
	  WriteFunctionBuffer("DECLOAKING");
	  break;
     }
}
			
SFDECL(funNavDoorStatus)
{
     switch (object->shipdata->door_status) {
		
     case DOORS_NONE:
	  WriteFunctionBuffer("NONE");
	  break;
     case DOORS_CLOSED:
	  WriteFunctionBuffer("CLOSED");
	  break;	
     case DOORS_OPEN:
	  WriteFunctionBuffer("OPEN");
	  break;
     }

}

SFDECL(funActive)
{
    FWriteFunctionBuffer("%d", (objFindObject(parseDbref(args[1])) == NULL)
			 ? 0 : 1);
}

SFDECL(funShip)
{
    FWriteFunctionBuffer("%d", Ship(object) ? 1 : 0);
}

SCDECL(cmdWarpTable)
{
     float setting;

     Notify(player, "Warp costs table:");
     Notify(player, " ");

     for (setting = 1.0; setting <= 10.0; setting++) 
	  FNotify(player, "Warp %3.1f costs %d.", setting, 
		  navWarpCost(object, setting));

}

SFDECL(funWarpCost)
{

    FWriteFunctionBuffer("%d", navWarpCost(object, atof(args[1])));
}

SFDECL(funMaxWarp)
{
    FWriteFunctionBuffer("%3.1f", navMaxWarp(object, atoi(args[1])));
}

SFDECL(funSPHtoXYZ)
{
     char *wbuf;
     SPH sph;
     XYZ xyz;

     wbuf = pse_malloc(SMALL_BUF_SIZE);
     sph.bearing = atof(args[1]);
     sph.elevation = atof(args[2]);
     sph.range = atof(args[3]);
     sph_to_xyz(sph, &xyz);

     sprintf(wbuf, RANGEF " " RANGEF " " RANGEF, xyz.x, xyz.y, xyz.z);
     WriteFunctionBuffer(wbuf);
     pse_free(wbuf);
}

SFDECL(funXYZtoSPH)
{
     char *wbuf;
     SPH sph;
     XYZ xyz;

     wbuf = pse_malloc(SMALL_BUF_SIZE);
     xyz.x = RANGEI(args[1]);
     xyz.y = RANGEI(args[2]);
     xyz.z = RANGEI(args[3]);

     xyz_to_sph(xyz, &sph);

     sprintf(wbuf, "%1.2f %+1.2f " RANGEF, sph.bearing, sph.elevation,
	     sph.range);
     WriteFunctionBuffer(wbuf);
     pse_free(wbuf);
}

SCDECL(cmdTacAutoCharge)
{
     SHIP *ship = object->shipdata;
     int i;

     ship->auto_online = !ship->auto_online;

     if (ship->auto_online) {

	  FNotify(player, "%s charging controls set to automatic.",
		  capstr(F_PTR(ship->rec, torp_string)));
	  for (i=0; i < F_INT(ship->rec, torp_qty); i++) {
	       if (ship->torp[i].status == TORP_LOADED) {
		    FNotify(player, "Bringing %s tube %d online:",
			    F_PTR(ship->rec, torp_string), i);
		    tacTorpControl(object, player, i, 1);
	       }
	  }

     }
     else
	  FNotify(player, "%s charging controls set to manual.",
		  capstr(F_PTR(ship->rec, torp_string)));

     return;
} 

SCDECL(cmdTacAutoReload)
{
     SHIP *ship = object->shipdata;
     int i;

     ship->auto_reload = !ship->auto_reload;

     if (ship->auto_reload) {
	  FNotify(player, "%s auto reloading enabled.", 
		  capstr(F_PTR(ship->rec, torp_string)));

	  for (i=0; i < F_INT(ship->rec, torp_qty) && 
		    ship->free_torp_loaders; i++) {
	       if (ship->torp[i].status == TORP_EMPTY) {
		    tacReloadTorp(object, player, i);
		    break;
	       }
	  }
     }
     else
	  FNotify(player, "%s auto reloading disabled.",
		  capstr(F_PTR(ship->rec, torp_string)));

     return;
}

SCDECL(cmdTacFireGun)
{
    XYZ pos1, pos2;
    int nfire, tdam, withc;
    int bank, damage = 0, fireall, i;
    TAG *target = NULL;
    SHIP *ship = object->shipdata;
    CONTACT *us;
    struct timeval now;
    
    fireall = strcmp(args[1], "all") ? 0 : 1;
    bank = atoi(args[1]);
    
    if (object->locked_on == NULL) {
	FNotify(player, "Weapons must be locked in order to fire %ss.",
		F_PTR(object->shipdata->rec, gun_string));
	return;
    }
    
    target = object->locked_on->listref;
    
    if (Pacifist(ship)) {
	Notify(player, "#-1 Weapon use is presently disabled.");
	return;
    }
    
    if (!fireall && (bank < 0 || bank >= F_INT(ship->rec, gun_qty))) {
	FNotify(player, "Invalid %s number.", F_PTR(ship->rec, gun_string));
	return;
    }
    
    /* nfire is the number of guns that fired. withc the number that
       were charged. */
    nfire = withc = 0;
    
    if (fireall) {
	for (i=0; i < F_INT(ship->rec, gun_qty); i++) 
	    if (ship->gun[i].charge > 0) {
		withc ++;
		tdam = tacFireGun(object, player, i);
		if (tdam >= 0) {
		    damage += tdam;
		    nfire ++;
		}
	    }
    } else {
	if (ship->gun[bank].charge > 0) {
	    withc ++;
	    tdam = tacFireGun(object, player, bank);
	    if (tdam >= 0) {
		damage = tdam;
		nfire ++;
	    }
	}
    }
    
    /* If no weapons were charged, we need to shout about it */
    if (withc == 0) {
	FNotify(player, "%s must be charged before firing.", 
		capstr(F_PTR(ship->rec, gun_string)));
	return;
    }
    
    /* We didn't actually fire any weapons - maybe cause we're disabled,
       or out of range, or something. The message already went out to
       the user about this problem in tacFireGun. */
    if (nfire == 0)
	return;
    
    /* At this point, we know at least 1 gun actually fired, and so we
       must do all the emits about who fired what at whom */
    us = snsFindContact(target, object);
    
    if (Ship(target)) {
	if (us != NULL) {
	    
	    MFNotify(target->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		     "[Incoming %s fire from contact [%d]: %s]",
		     F_PTR(ship->rec, gun_string), us->contact_number,
		     object->rec->name);
	    
	} else {
	    MFNotify(target->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		     "[Incoming %s fire from an unknown source]",
		     F_PTR(ship->rec, gun_string));
	}
    }
    
    /* Send the attacked events */
    evTrigger(target, EVENT_ATTACKED, "i",
	      (us == NULL) ? -1 : us->contact_number);
    evBroadcast(object, target, "fired upon", EVENT_ATTACKED_OTHER);
    
    gettimeofday(&now, NULL);
    
    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);
    
    /* Log if the object is in a space so marked. */
    if (space_info[object->space].flags & SPACE_LOGGED)
	log_space("%s (#%d) fired guns at %s (#%d) for %d damage from "
		  "range " RANGEF ".", object->rec->name, 
		  object->rec->db_obj,
		  target->rec->name, target->rec->db_obj, damage,
		  distance(&pos1, &pos2));
    
    if (damage > 0)
	damBattleDamage(pos1, target, damage, WPN_GUN);
    
    return;
}

SCDECL(cmdTacFireTorp)
{
    XYZ pos1, pos2;
    TAG *target = NULL;
    int tube;
    int hits = 0, fired = 0, damage = 0, i;
    int canshoot = 0;
    int fireall, nolock;
    SHIP *ship = object->shipdata;
    CONTACT *us;
    SPH direction;
    XYZ rel_pos;
    struct timeval now;
    
    fireall = strcmp(args[1], "all") ? 0 : 1;
    tube = atoi(args[1]);
    
    if (strcmp(args[2], "NONE") != 0) {
	nolock = 1;
	direction.bearing = atof(args[2]);
	direction.elevation = atof(args[4]);
	
	if (strlen(args[4]) && *(args[3])=='-')
	    direction.elevation = -direction.elevation;
	direction.range = atoi(args[5]) / F_FLOAT(object->rec, range_factor);
    }
    else {
	nolock = 0;
	direction.range = 0;
    }
    
    if (Cloaked(object)) {
	Notify(player, "You cannot fire while cloaked.");
	return;
    }
    
    if (Pacifist(object->shipdata)) {
	Notify(player, "OOC:  Weapon use is presently disabled.");
	return;
    }
    
    if (!fireall && (tube < 0 || tube >= F_INT(ship->rec, torp_qty))) {
	FNotify(player, "Invalid %s number.", F_PTR(ship->rec, torp_string));
	return;
    }
    
    if (fireall) {
	for (i=0; i < F_INT(ship->rec, torp_qty); i++)
	    if (ship->torp[i].status == TORP_ARMED)
		canshoot = 1;
	
	if (!canshoot) {
	    FNotify(player, "At least one %s must be armed before "
		    "firing.", F_PTR(ship->rec, torp_string));
	    return;
	}
    }
    else {
	if (ship->torp[tube].status != TORP_ARMED) {
	    FNotify(player, "%s must be armed before firing.", 
		    capstr(F_PTR(ship->rec, torp_string)));
	    return;
	}
    }
    
    if (object->locked_on == NULL) {
	if (!nolock) {
	    FNotify(player, "A heading must be specified if %ss "
		    "are to be fired without lock.", 
		    F_PTR(object->shipdata->rec, torp_string));
	    return;
	}
	
    }
    
    gettimeofday(&now, NULL);
    
    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);

    if (!nolock) {
	target = object->locked_on->listref;
	rel_pos.x = pos2.x - pos1.x;	
	rel_pos.y = pos2.y - pos1.y;	
	rel_pos.z = pos2.z - pos1.z;
	xyz_to_sph(rel_pos, &direction);
	direction.range = 0;
    }
    
    tacInitFireTable(object, direction);
    
    if (fireall) {
	for (i=0; i < F_INT(ship->rec, torp_qty); i++) 
	    if (ship->torp[i].status == TORP_ARMED) {
		fired++;
		if (tacFireTorp(object, player, i, direction, nolock) > 0) {
		    hits++;
		    damage += ship->torp[i].power;
		}
	    }
    }
    else {
	fired = 1;
	if (tacFireTorp(object, player, tube, direction, nolock)) {
	    damage += ship->torp[tube].power;
	    hits++;
	}
    }
    
    tacFreeFireTable();
    
    /* No messages or events if no torps were fired. This should never
       happen due to the checking earlier on. */
    if (fired == 0)
	return;
    
    if (!nolock) {
	us = snsFindContact(target, object);
	
	if (Ship(target)) {
	    if (us != NULL) {
		MFNotify(target->shipdata, -1, CONS_TAC, CONS_ACTIVE,
			 "[Incoming %s%s from contact [%d]: %s]",
			 F_PTR(ship->rec, torp_string), 
			 fireall ? "s fired" : " fired", 
			 us->contact_number, object->rec->name);
	    } else {
		MFNotify(target->shipdata, -1, CONS_TAC, CONS_ACTIVE,
			 "[Incoming %s%s from an unknown source]",
			 F_PTR(ship->rec, gun_string), 
			 fireall ? "s fired" : " fired");
	    }
	}
	
	evTrigger(target, EVENT_ATTACKED, "i", (us == NULL) ? -1 :
		  us->contact_number);
	evBroadcast(object, target, "fired upon", 
		    EVENT_ATTACKED_OTHER);
	
    }
    else
	evBroadcast(object, NULL, "fired upon an unknown target",
		    EVENT_ATTACKED_OTHER);
    
    if (!nolock) {
	if (Ship(target))
	    MFNotify(target->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		     "[%d/%d enemy %ss hit]", hits, fireall ? fired : 1,
		     F_PTR(ship->rec, torp_string));
	
	if (space_info[object->space].flags & SPACE_LOGGED)
	    log_space("%s (#%d) hit %s (#%d) with %d/%d torps for "
		      "%d damage from range " RANGEF ".", 
		      object->rec->name, 
		      object->rec->db_obj, target->rec->name, 
		      target->rec->db_obj, hits, fired, damage, 
		      distance(&pos1, &pos2));
	
	if (hits > 0)
	    damBattleDamage(pos1, target, damage, WPN_TORP);
    }
    
    return;
}

SCDECL(cmdTacLock)
{
     tacLockWeapons(object, player, atoi(args[1]));	
}

SCDECL(cmdTacUnlock)
{
     tacUnlock(object, player);
}

SCDECL(cmdTacTractorLock)
{
     tacTractorLock(object, player, atoi(args[1]));	
}

SCDECL(cmdTacTractorUnlock)
{
     tacTractorUnlock(object, player);
}

SCDECL(cmdTacTractorDock)
{
     SHIP *ship = object->shipdata;

     switch (ship->tractor_status) {
     case TRACTOR_NONE:
	  Notify(player, "This ship is not equipped with a tractor beam "
		 "generator.");
	  break;

     case TRACTOR_OFF:
     case TRACTOR_DAMAGED:
	  Notify(player, "The tractor beam is not presently engaged.");
	  break;

     default:
	  Notify(player, "Attempting to pull the tractor beam target into"
		 " dock.");
	  navDock(object->shipdata->tractor_target, player, object, 1);
	  break;
     }
}

SCDECL(cmdTacTractorDraw)
{
     SHIP *ship = object->shipdata;
     int action;

     action = atoi(args[1]);

     switch (ship->tractor_status) {
     case TRACTOR_NONE:
	  Notify(player, "This ship is not equipped with a tractor beam "
		 "generator.");
	  break;

     case TRACTOR_OFF:
     case TRACTOR_DAMAGED:
	  Notify(player, "The tractor beam is not presently engaged.");
	  break;

     case TRACTOR_ON:	
	  if (action) {
	       Notify(player, "Tractor beam draw mode engaged.");
	       ship->tractor_status = TRACTOR_DRAW;
	  }
	  else
	       Notify(player, "Tractor beam draw mode is not "
		      "engaged.");
	  break;

     case TRACTOR_DRAW:
	  if (!action) {
	       Notify(player, "Tractor beam draw mode disengaged.");
	       ship->tractor_status = TRACTOR_ON;
	  }
	  else
	       Notify(player, "Tractor beam draw mode is already "
		      "engaged.");
	  break;
     }

     return;
}

SCDECL(cmdTacTorpOnline)
{
     tacTorpControl(object, player, atoi(args[1]), 1);
}

SCDECL(cmdTacTorpOffline)
{
     tacTorpControl(object, player, atoi(args[1]), 0);
}

SCDECL(cmdTacGunOnline)
{
     tacGunControl(object, player, atoi(args[1]), 1);
}

SCDECL(cmdTacGunOffline)
{
     tacGunControl(object, player, atoi(args[1]), 0);
}

SCDECL(cmdTacReloadTorp)
{
     tacReloadTorp(object, player, atoi(args[1]));
}

SCDECL(cmdTacAllocate)
{
     tacAllocate(object, player, atoi(args[1]), atoi(args[2]), 
		 atoi(args[3]));
}

SFDECL(funScan)
{
     snsScan(object, mud, atoi(args[1]));
}

SFDECL(funTacAllocGuns)
{
     FWriteFunctionBuffer("%d", object->shipdata->talloc_guns);
}

SFDECL(funTacAllocTorps)
{
     FWriteFunctionBuffer("%d", object->shipdata->talloc_torps);
}

SFDECL(funTacAllocTractor)
{
     FWriteFunctionBuffer("%d", object->shipdata->talloc_tractor);
}

SFDECL(funTacGunQty)
{
     FWriteFunctionBuffer("%d", F_INT(object->shipdata->rec, gun_qty));
}

SFDECL(funTacGunStatus)
{
     int gun;

     gun = atoi(args[1]);
     if (gun >= 0 && gun < F_INT(object->shipdata->rec, gun_qty)) 
	  switch (object->shipdata->gun[gun].status) {
	  case GUN_ONLINE:
	       WriteFunctionBuffer("ON");
	       break;	
	  case GUN_OFFLINE:
	       WriteFunctionBuffer("OFF");
	       break;	
	  case GUN_DAMAGED:
	       WriteFunctionBuffer("DAMAGED");
	       break;	
	  }
}

SFDECL(funTacGunCharge)
{
     int gun;

     gun = atoi(args[1]);
     if (gun >= 0 && gun < F_INT(object->shipdata->rec, gun_qty)) 
	  FWriteFunctionBuffer("%d", 
			       object->shipdata->gun[gun].charge);

}

SFDECL(funTacGunPower)
{
     int gun;

     gun = atoi(args[1]);
     if (gun >= 0 && gun < F_INT(object->shipdata->rec, gun_qty)) 
	  FWriteFunctionBuffer("%d", 
			       object->shipdata->gun[gun].power);

}

SFDECL(funTacTorpQty)
{
     FWriteFunctionBuffer("%d", F_INT(object->shipdata->rec, torp_qty));
}

SFDECL(funTacTorpAmmo)
{
     FWriteFunctionBuffer("%d", F_INT(object->shipdata->rec, torp_ammo));
}

SFDECL(funTacTorpStatus)
{
     int tube;

     tube = atoi(args[1]);
     if (tube >= 0 && tube < F_INT(object->shipdata->rec, torp_qty)) 
	  switch (object->shipdata->torp[tube].status) {
	  case TORP_EMPTY:
	       WriteFunctionBuffer("EMPTY");
	       break;	
	  case TORP_LOADING:
	       WriteFunctionBuffer("RELOADING");
	       break;	
	  case TORP_LOADED:
	       WriteFunctionBuffer("LOADED");
	       break;	
	  case TORP_ONLINE:
	       WriteFunctionBuffer("CHARGING");
	       break;	
	  case TORP_ARMED:
	       WriteFunctionBuffer("ARMED");
	       break;	
	  case TORP_DAMAGED:
	       WriteFunctionBuffer("DAMAGED");
	       break;	
	  }
}

SFDECL(funTacTorpCharge)
{
     int tube;

     tube = atoi(args[1]);
     if (tube >= 0 && tube < F_INT(object->shipdata->rec, torp_qty)) 
	  FWriteFunctionBuffer("%d", 
			       object->shipdata->torp[tube].charge);

}

SFDECL(funTacTorpPower)
{
     int tube;

     tube = atoi(args[1]);
     if (tube >= 0 && tube < F_INT(object->shipdata->rec, torp_qty)) 
	  FWriteFunctionBuffer("%d", 
			       object->shipdata->torp[tube].power);

}

SFDECL(funTacLockStatus)
{
     if (object->locked_on) 
	  FWriteFunctionBuffer("LOCKED %d", 
			       object->locked_on->contact_number);
     else if (object->pending_lock)
	  FWriteFunctionBuffer("PENDING %d",
			       object->pending_lock->contact_number);
     else
	  WriteFunctionBuffer("OFF");
}

SFDECL(funSensorStatus)
{
     if (object->shipdata->damage[SYSTEM_SENSORS].status > '6')
	  WriteFunctionBuffer("DAMAGED");
     else
	  WriteFunctionBuffer("OK");
}
		
SFDECL(funScannerStatus)
{
     if (object->shipdata->damage[SYSTEM_SCANNERS].status > '6')
	  WriteFunctionBuffer("DAMAGED");
     else
	  WriteFunctionBuffer("OK");
}

SFDECL(funTacReloadStatus)
{
     if (object->shipdata->auto_reload)
	  WriteFunctionBuffer("ON");
     else
	  WriteFunctionBuffer("OFF");
}

SFDECL(funTacChargeStatus)
{
     if (object->shipdata->auto_online)
	  WriteFunctionBuffer("ON");
     else
	  WriteFunctionBuffer("OFF");
}

SFDECL(funTacTractorTarget)
{
     CONTACT *contact;

     contact = snsFindContact(object, object->shipdata->tractor_target);

     if (contact != NULL)
	  FWriteFunctionBuffer("%d", contact->contact_number);

}

SFDECL(funTacTractorStatus)
{
     SHIP *ship = object->shipdata;

     switch (ship->tractor_status) {

     case TRACTOR_NONE:
	  WriteFunctionBuffer("NONE");
	  break;	
     case TRACTOR_OFF:
	  WriteFunctionBuffer("OFF");
	  break;	
     case TRACTOR_ON:
	  WriteFunctionBuffer("ON");
	  break;	
     case TRACTOR_DRAW:
	  WriteFunctionBuffer("DRAW");
	  break;
     case TRACTOR_DAMAGED:
	  WriteFunctionBuffer("DAMAGED");
	  break;
     }
     return;
}

SCDECL(cmdCauseDamage)
{
    XYZ pos1, pos2;
    TAG *target;
    int damage;
    int type;
    const char *dstr;
    struct timeval now;
    
    target = objFindObject(parseDbref(args[1]));
    damage = atoi(args[2]);
    type = atoi(args[3]);
    
    if (target==NULL) {
	Notify(player, "Invalid target.");
	return;
    }
    
    gettimeofday(&now, NULL);
	 
    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);
    
    if (space_info[object->space].flags && SPACE_LOGGED) {
	
	switch (type) {
	case 0: 
	    dstr="gun";
	    break;
	case 1: 
	    dstr="torp";
	    break;
	case 2: 
	    dstr="explosion";
	    break;
	case 3: 
	    dstr="";
	    break;
	default:
	    dstr="<undefined damage type>";
	    break;
	}
	
	log_space("%s (#%d) hits %s (#%d) with %d points of %s damage "
		  "from range " RANGEF ".", object->rec->name, 
		  object->rec->db_obj, target->rec->name, 
		  target->rec->db_obj, damage, dstr, 
		  distance(&pos1, &pos2));
	
    }
    
    damBattleDamage(pos1, target, atoi(args[2]), atoi(args[3]));
}

SCDECL(cmdCauseSystemDamage)
{

     int system;
     int points;

     system = atoi(args[1]);
     points = atoi(args[2]);

     if ((system < 0) || (system >= NUM_SYSTEMS)) {
	  Notify(player, "Invalid system to damage.");
	  return;
     }

     if (points < 0) {
	  Notify(player, "Invalid amount to damage system by.");
	  return;
     }

     if (space_info[object->space].flags & SPACE_LOGGED) {
	  log_space("%s (#%d) suffered %d points of system damage to "
		    "system %s (%d).", object->rec->name, 
		    object->rec->db_obj, points, "something", system); 
     }

     damSystemDamage(object, system, points);

}

SCDECL(cmdExplosion)
{

     int damage, space;
     XYZ point;

     space = atoi(args[1]);

     if (space < 0 || space >= NUM_SPACES) {
	  FNotify(player, "Valid spaces are between 0 and %d.",
		  NUM_SPACES - 1);
	  return;
     }
		
     point.x = atoi(args[2]);
     point.y = atoi(args[3]);
     point.z = atoi(args[4]);
     damage = atoi(args[5]);

     damExplosionDamage(space, point, damage);

     return;
}

SCDECL(cmdDrainReactor)
{
     SHIP *ship = object->shipdata;
     int power;
     int old_drain;

     power = atoi(args[1]);

     old_drain = object->shipdata->reactor_drain;

     if (power >= 0) {
	  object->shipdata->reactor_drain = power;

	  MFNotify(object->shipdata, player, CONS_ENG, CONS_ACTIVE,
		   "[Engine power %s.  New maximum output is %d]",
		   power > old_drain ? "draining" : "restored",
		   ship->current_reactor_output_max - ship->reactor_drain);

	  engAllocCheck(object);
	  evTrigger(object, EVENT_REACTOR_DRAIN, "");
     }

     return;
}	

SFDECL(funName)
{
     FWriteFunctionBuffer("%s", object->rec->name);
}

SFDECL(funListContainers)
{
     char *buff;
     
     buff = pse_malloc(MAX_ATTRIBUTE_LEN);
     evListContainers(object, buff);
     WriteFunctionBuffer(buff);
     pse_free(buff);
}

SCDECL(cmdSetTorpAmmo)
{
     F_INT(object->shipdata->rec, torp_ammo) = atoi(args[1]);
}

SCDECL(cmdCommSetChannel)
{
     int channel;
     unsigned int freq;

     channel = atoi(args[1]);
     freq = atoi(args[2]);

     if (channel < 0 || channel >= MAX_COMM_CHANNELS) {
	  FNotify(player, "Valid channels are between 0 and %d inclusive.",
		  MAX_COMM_CHANNELS-1);
	  return;
     }

     if (freq < 0 || freq > 65536) {
	  Notify(player, "Valid frequencies are between 1 and 65536.");
	  return;
     }

     commSetChannel(object, player, channel, freq);

     return;
}

SCDECL(cmdCommLabelChannel)
{
     int channel;

     channel = atoi(args[1]);

     if (channel < 0 || channel >= MAX_COMM_CHANNELS) {
	  FNotify(player, "Valid channels are between 0 and %d inclusive.",
		  MAX_COMM_CHANNELS-1);
	  return;
     }

     commLabelChannel(object, player, channel, args[2]);

     return;
}

SFDECL(funCommSend)
{
    int channel;
    
    channel = atoi(args[1]);
    
    if (channel < 0 || channel >= MAX_COMM_CHANNELS) {
	
	FWriteFunctionBuffer("#-1 Valid channels are between 0 and %d "
			     "inclusive.", MAX_COMM_CHANNELS-1);
	return;
    }
    
    commSend(object, mud, channel, args[2]);
    
    return;
}

SFDECL(funCommFreq)
{
     int channel;

     channel = atoi(args[1]);

     if (channel < 0 || channel >= MAX_COMM_CHANNELS) {
	 FWriteFunctionBuffer2("#-1 INVALID CHANNEL %d (0-%d)", channel,
			      MAX_COMM_CHANNELS);
	 return;
     }
     
     FWriteFunctionBuffer("%d",object->shipdata->channel[channel].frequency);

     return;
}

SFDECL(funCommLabel)
{
     int channel;

     channel = atoi(args[1]);

     if (channel < 0 || channel >= MAX_COMM_CHANNELS) {
	 FWriteFunctionBuffer2("#-1 INVALID CHANNEL %d (0-%d)", channel,
			      MAX_COMM_CHANNELS);
	 return;
     }
     
     WriteFunctionBuffer(object->shipdata->channel[channel].label);

     return;
}

SCDECL(cmdDamRepair)
{
     int team;

     team = atoi(args[1]);

     /* valid team check */
     if (team < 0 || team >= F_INT(object->shipdata->rec, damcon_teams)) {
	  Notify(player, "Invalid team number.");
	  return;
     }

     damRepair(object, player, team, args[2]);
     return;
}

SCDECL(cmdDamSystemStatus)
{
     damSystemStatus(object, player);
}

SCDECL(cmdDamTeamStatus)
{
     damTeamStatus(object, player);
}

SFDECL(funContactListNumbers)
{
     char *buff;
     
     buff = pse_malloc(MAX_ATTRIBUTE_LEN);
     snsListContacts(buff, object->contact_list, 0, 0);
     WriteFunctionBuffer(buff);
     pse_free(buff);
}

SFDECL(funContactListDbrefs)
{
     char *buff;
    
     buff = pse_malloc(MAX_ATTRIBUTE_LEN);
     snsListContacts(buff, object->contact_list, 0, 1);
     WriteFunctionBuffer(buff);
     pse_free(buff);
}

SFDECL(funContactInfoLevel)
{
     CONTACT *contact;

     contact = snsFindContactByNumber(object, atoi(args[1]));

     if (contact != NULL)
	  FWriteFunctionBuffer("%d", contact->info_level);
     else
	  WriteFunctionBuffer("0");
}

SFDECL(funTransListBeamable)
{
     char *buff;
     
     buff = pse_malloc(MAX_ATTRIBUTE_LEN);
     snsListContacts(buff, object->contact_list, 2, 1);
     WriteFunctionBuffer(buff);
     pse_free(buff);
}

SFDECL(funTransListDests)
{
     char *buff;
     
     buff = pse_malloc(MAX_ATTRIBUTE_LEN);
     snsListContacts(buff, object->contact_list, 1, 1);
     WriteFunctionBuffer(buff);
     pse_free(buff);
}

SFDECL(funTransPermission)
{
    XYZ pos1, pos2;
    TAG *target;
    struct timeval now;
    
    if (Ship(object))
	if (object->shipdata->damage[SYSTEM_TRANSPORTER].status >= 
	    '7') {
	    WriteFunctionBuffer("DAMAGED");
	    return;
	}
    
    if ((target=objFindObject(parseDbref(args[1])))==NULL) {
	WriteFunctionBuffer("INVALID TARGET");
	return;
    }
    
    if (!CanBeamTo(target)) {
	WriteFunctionBuffer("INVALID TARGET");
	return;
    }
    
    if (!CanBeamFrom(target)) {
	WriteFunctionBuffer("INVALID SOURCE");
	return;
    }
    
    gettimeofday(&now, NULL);
    
    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);
    
    if (distance(&pos1, &pos2) > 
	F_INT(object->rec, xport_range)) {
	WriteFunctionBuffer("OUT OF RANGE");
	return;
    }
    
    if (Ship(object)) {
	if (object->shipdata->shield_status[calcFacingShield(pos2, object)] ==
	    SHLD_UP) {
	    WriteFunctionBuffer("SOURCE SHIELD");
	    return;
	}
    }
    
    if (Ship(target)) {
	if (target->shipdata->shield_status[calcFacingShield(pos1, target)] ==
	    SHLD_UP) {
	    WriteFunctionBuffer("TARGET SHIELD");
	    return;
	}
    }
    
    WriteFunctionBuffer("OK");
    return;
}

SFDECL(funRange)
{
    TAG *target;
    XYZ pos1, pos2;
    struct timeval now;
    
    if ((target=objFindObject(parseDbref(args[1])))==NULL) {
	WriteFunctionBuffer("#-1 Invalid target");
	return;
    }	
    
    gettimeofday(&now, NULL);
    
    object_position(object, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);    
    
    FWriteFunctionBuffer("%.f",
			 distance(&pos1, &pos2) *
			 F_FLOAT(object->rec, range_factor));
}

SFDECL(funOwnerNumber)
{
     FWriteFunctionBuffer("%d", F_INT(object->shipdata->rec, owner_num));
}

SFDECL(funFacingShield)
{
    TAG *target;
    XYZ pos;
    struct timeval now;
    
    if ((target=objFindObject(parseDbref(args[1])))==NULL) {
	WriteFunctionBuffer("#-1 Invalid target");
	return;
    }
    
    gettimeofday(&now, NULL);
    
    object_position(target, &pos, &now, 0);
    FWriteFunctionBuffer("%d", calcFacingShield(pos,object));
}

SFDECL(funFormatRangeString)
{
     TAG *target;
     char *stage;
     
     target = objFindObject(parseDbref(args[1]));

     stage = pse_malloc(SMALL_BUF_SIZE);
     navRangeString(target, RANGEI(args[2]), stage, atoi(args[3]));
     WriteFunctionBuffer(stage);
     pse_free(stage);
}

/* funMetricScale
 *
 * Arg1: Number to convert to a scaled metric format
 * Arg2: Width of output. Must be 4, or more than 5.
 * Arg3: Output formatting options.
 *   bit0=1: Replace any leading - with a space
 *   bit1=1: Replace any leading + with a space
 *   bit2=1: Remove any leading space
 */
SFDECL(funMetricScale)
{
     char         *stage;
     float        value;
     unsigned int width, options;

     value	= atof(args[1]);
     width	= atoi(args[2]);
     options	= atoi(args[3]);

     stage = pse_malloc(SMALL_BUF_SIZE);

     /* Do the conversion */
     MetricScale(value, width, options, stage);

     /* Write out the result */
     WriteFunctionBuffer(stage);
     pse_free(stage);
}

/* funListModules
 *
 * List the names of the specified module class.
 * Currently permitted options are:
 *
 * range:  Show the range translation modules
 * shield: Show the shield plugin modules (optional)
 */
SFDECL(funListModules)
{
    if (strcmp("range", args[1]) == 0) {
	WriteFunctionBuffer(rxlListRegisteredModules());
	return;
    }

#ifdef ENABLE_SHIELD_CLASSES
    if (strcmp("shield", args[1]) == 0) {
	WriteFunctionBuffer(scmListRegisteredModules());
	return;
    }
#endif
    
    WriteFunctionBuffer("#-1 Unknown module specified.");
}

/* funSpaceOptions
 *
 * Return a list of the optional components compiled into to PSE, along with
 * a list of shield plugins.
 */
SFDECL(funSpaceOptions)
{
    SSTR *sstr;
    char *buff;
    
    sstr = sstr_new(MAX_ATTRIBUTE_LEN);
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    sstr_cat(sstr,
	     "Portable Space Engine Options\n"
	     "-----------------------------\n\n", 0);
    
    sprintf(buff, "Number of spaces:         %d\n", NUM_SPACES);
    sstr_cat(sstr, buff, 0);
    
    sprintf(buff, "Maximum attribute length: %d\n", MAX_ATTRIBUTE_LEN);
    sstr_cat(sstr, buff, 0);
    
    sprintf(buff, "Debug output object:      %d\n", debug_char);
    sstr_cat(sstr, buff, 0);
    
#ifdef ENABLE_ODOMETER
    sstr_cat(sstr, "Ship odometers:           Enabled\n", 0);
#else
    sstr_cat(sstr, "Ship odometers:           Disabled\n", 0);
#endif

#ifdef ENABLE_TIME_TRACKER
    sstr_cat(sstr, "Ship time tracking:       Enabled\n", 0);
#else
    sstr_cat(sstr, "Ship time tracking:       Disabled\n", 0);
#endif

#ifdef ENABLE_FLOATS
    sstr_cat(sstr, "Coordinate Scheme:        Floating Point\n", 0);
#else
#ifdef ENABLE_ENABLE_LONG_LONG_COORDS
    sstr_cat(sstr, "Coordinate Scheme:        Long-Long (depreciated)\n", 0);
#else
    sstr_cat(sstr, "Coordinate Scheme:        Standard Integer\n", 0);
#endif
#endif

    sprintf(buff, "Range Plugins:            %s\n\n",
	    rxlListRegisteredModules());
    sstr_cat(sstr, buff, 0);
    
    sstr_cat(sstr,
	     "Non-Standard Features\n"
	     "---------------------\n\n", 0);
    
#ifdef ENABLE_SHIELD_CLASSES
    sstr_cat(sstr, "Partial:   Shield class support code installed.\n", 0);
#endif

    pse_free(buff);
    WriteFunctionBuffer(sstr_str(sstr));
    sstr_free(sstr);
}

SFDECL(funSetSpaceName)
{
     int space;

     space = atoi(args[1]);

     if (space < 0 || space >= NUM_SPACES) {
	  FWriteFunctionBuffer2("#-1 Invalid space (%d). "
		  "NUM_SPACES is set to %d.", space, NUM_SPACES);
	  return;
     }

     if (strlen(args[2]) >= MAX_SPACE_NAME_LEN) {
	  FWriteFunctionBuffer("The maximum length of the name of a "
			       "space is %d characters.",
			       MAX_SPACE_NAME_LEN-1);
	  return;
     }

     strcpy(space_info[space].name, args[2]);
}

SFDECL(funSetDebugChar)
{
     dbref dbc;

     dbc = parseDbref(args[1]);
     
     if (ValidObject(dbc)) {
	 FWriteFunctionBuffer("Debug character set to #%d.", dbc);
	 debug_char = dbc;
     } else
	 FWriteFunctionBuffer("#-1 That object (#%d) is not valid.", dbc);

     return;
}

SFDECL(funGetDebugChar)
{
     FWriteFunctionBuffer("#%d", debug_char);
}

SFDECL(funGetSpaceName)
{
     int space;

     space = atoi(args[1]);

     if (space < 0 || space >= NUM_SPACES) {
	  FWriteFunctionBuffer2("#-1 INVALID SPACE %d (0-%d)", space,
			      NUM_SPACES);
	  return;
     }

     WriteFunctionBuffer(space_info[space].name);
}

SFDECL(funSensorSanity)
{
#ifdef ENABLE_DEBUGGING
     int space;
	
     space = atoi(args[1]);

     if (space < 0 || space >= NUM_SPACES) {
	  FWriteFunctionBuffer("Valid spaces are between 0 and %d.",
		  NUM_SPACES - 1);
	  return;
     }

     snsSanityCheck(space);
     
     WriteFunctionBuffer("Any detected oddities have been logged.");
     
     return;
#else
     WriteFunctionBuffer("#-1 Debugging is not enabled.");
#endif
}

SFDECL(funAddTimer)
{
    int first, interval, count;
    
    first = atoi(args[1]);
    interval = atoi(args[2]);
    count = atoi(args[3]);
    
    if (first < 0) {
	WriteFunctionBuffer("#-1 Timers must have an initial interval >= 0");
	return;
    }
    
    if (interval < 0) {
	WriteFunctionBuffer("#-1 Timers must have a intervals >= 0");
	return;
    }
    
    if (count < 0) {
	WriteFunctionBuffer("#-1 Timers must have a count >= 0");
	return;
    }
    
    add_timer(object, first, interval, count, args[4]);
}

SFDECL(funDelTimer)
{
    if (*args[1] == '\0')
	del_timer(object, NULL);
    else
	del_timer(object, args[1]);
}

SFDECL(funListTimers)
{
    list_timers(object, mud);
}

#ifdef ENABLE_TURN_BASED
SFDECL(funRunCycle)
{
    int space;

    space = atoi(args[1]);
    
    if (space == -1) {
	
	for (space = 0; space < NUM_SPACES; space ++)
	    space_info[space].flags |= SPACE_DO_TURN;
	
	return;
    }
    
    if (space < 0 || space >= NUM_SPACES) {
	FWriteFunctionBuffer("#-1 Invalid space (%d).", space);
	return;
    }
    
    space_info[space].flags |= SPACE_DO_TURN;
}
#endif
