/* v0.9
 *
 * nav.c:  Ship navigation code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "nav.h"
#include "damage.h"
#include "sensors.h"
#include "smisc.h"
#include "events.h"
#include "shields.h"

typedef struct divisor_entry {

     const char *unit;		/* Unit name */
     float	scale;		/* Basic divisor */
     float	min;		/* Smallest value at this scale */
     float	max;		/* Largest value at this scale */
} SCALER;

SCALER range_divisors[] = {

     {"", 1.0, 1.0, 1000.0},
     {"K", 1000.0, 1000.0, 1000000.0},
     {"M", 1000000.0, 1000000.0, 1000000000.0},
     {"G", 1000000000.0, 1000000000.0, -1},

/* List terminator */
     {"", 0.0, -1, -1}
};

SCALER range_multipliers[] = {

     {"", 1.0, 0.001, 1.0},
     {"m", 1000.0, 0.000001, 0.001},
     {"u", 1000000.0, 0.000000001, 0.000001},
     {"n", 1000000000.0, -1, 0.000000001},

/* List terminator */
     {"", 0.0, -1, -1}
};

void navSetCourse(TAG *object, int player, float bearing, float elevation)
{
     /* check for legal values */
     if (bearing > 360 || bearing < 0) {
	  Notify(player, "Valid headings are from 0 to 360 inclusive.");
	  return;
     }

     if (bearing == 360)
	bearing = 0;

     if (elevation > 90 || elevation < -90) {
	  Notify(player, "Valid elevation/depressions are from -90 to "
		 "90 inclusive.");
	  return;
     }

     /* set desired change in heading */

     object->heading_adj.bearing = bearing - object->heading.bearing;

     if (object->heading_adj.bearing < -180)
	  object->heading_adj.bearing += 360.0;

     else if (object->heading_adj.bearing > 180)
	  object->heading_adj.bearing -= 360.0;

     object->heading_adj.elevation = elevation - object->heading.elevation;

     if (object->heading_adj.elevation != 0 || 
	 object->heading_adj.bearing != 0)
	  MFNotify(object->shipdata, player, CONS_NAV, CONS_ACTIVE,
		   "[Turning to %3.2f%+3.2f]", bearing, elevation);
     else
	  MFNotify(object->shipdata, player, CONS_NAV, CONS_ACTIVE,
		   "[Now heading %3.2f%+3.2f]", bearing, elevation);

     return;

}

void navAllocCheck(TAG *object)
{
     SHIP *ship = object->shipdata;
     float allowed_speed;
     dbref navigator;

     navigator = -1;

     if (navWarpCost(object, ship->warp_set_speed) > ship->alloc_nav) {

	  allowed_speed = navMaxWarp(object, ship->alloc_nav);
	  if (allowed_speed < 1.0)
	       allowed_speed = 0.0;
	  navSetWarp(object, navigator, allowed_speed);

     }
     
     return;
}

int navWarpCost(TAG *object, float speed)
{
     if (speed > 0.0 && !Speedy(object->shipdata))
	  return ((int)(290 + 10 * speed * speed) * 
		  object->shipdata->warp_factor *
		  (Pokey(object->shipdata) ? 2.0 : 1.0));
     else
	  return 0;
}

float navMaxWarp(TAG *object, int cost)
{
    float warpsq, sqt;
    
    if (Pokey(object->shipdata))
	cost = cost / 2;	
    
    warpsq = ((float) cost)/(10.0 * object->shipdata->warp_factor) - 29.0;
    
    if (Speedy(object->shipdata))
	return object->shipdata->warp_rated_max;

    if (warpsq < 1.0)
	return 0.0;
    
    sqt = sqrt(warpsq);
    
    if (sqt > object->shipdata->warp_rated_max)
	return object->shipdata->warp_rated_max;
    
    return sqt;
}

void navSetWarp(TAG *object, dbref player, float setting)
{

     SHIP *ship = object->shipdata;
     int energy_req;

     if (!CanMove(object)) {
	  Notify(player, "This object cannot move.");
	  return;
     }

     if (setting == 0.0) {
	  MFNotify(ship, player, CONS_NAV, CONS_ACTIVE, "[Warp disengaged]");
	  ship->warp_set_speed = 0.0;
	  return;
     }

     if (setting < 1.0) {

#ifndef ENABLE_FLOATS
	  Notify(player, "Warp setting must be over 1.0.");
	  return;
#else
	  /* Very preliminary impulse/sublight stuff.
	   * TODO: energy check, top speed check */
	  MFNotify(ship, player, CONS_NAV, CONS_ACTIVE, "[Speed set at %3.2f%% impulse]",
		   setting * 100);
          ship->warp_set_speed = setting;
	  return;
#endif
     }

     if (setting > ship->warp_rated_max) {
	  FNotify(player, "Engines on this ship are not capable of"
		  " speeds in excess of warp %3.1f.", ship->warp_rated_max);
	  return;
     }

     energy_req = navWarpCost(object, setting);

     if (Speedy(ship))
	  energy_req = 0;

     if (ship->alloc_nav < energy_req) {
	  FNotify(player, "Not enough energy allocated to drives for"
		  " warp %3.1f.", setting);
	  FNotify(player,"Power required for that setting is %d.",
		  energy_req);
	  return;
     }

     ship->warp_set_speed = setting;
     MFNotify(ship, player, CONS_NAV, CONS_ACTIVE, "[Speed set at %3.1f]",
	      setting);

     return;
}

void navDoorControl(TAG *object, dbref player, int action)
{
     SHIP *ship = object->shipdata;

     switch (ship->door_status) {

     case DOORS_NONE:
	  Notify(player, "This ship has no docking bay doors.");
	  break;

     case DOORS_CLOSED:
	  if (!action) 
	       Notify(player, "Docking bay doors are already closed.");
	  else {
	       ship->door_status = DOORS_OPEN;
	       MFNotify(ship, player, CONS_NAV, CONS_ACTIVE,
			"[Docking bay doors opened]");
	       evTrigger(object, EVENT_BAY_DOORS, "i", 1);
	  }
	  break;

     case DOORS_OPEN:
	  if (action) 
	       Notify(player, "Docking bay doors are already open.");
	  else {
	       ship->door_status = DOORS_CLOSED;
	       MFNotify(ship, player, CONS_NAV, CONS_ACTIVE,
			"[Docking bay doors closed]");
	       evTrigger(object, EVENT_BAY_DOORS, "i", 0);
	  }
	  break;

     }

     return;
}

void navLand(TAG *object, dbref player, int target)
{
     CONTACT *planet;
     char buff[SMALL_BUF_SIZE];

     planet = snsFindContactByNumber(object, target);

     if (planet==NULL) {
	  Notify(player, "Invalid target.");
	  return;
     }

     if (!CanLand(object->shipdata)) {
	  Notify(player, "This ship cannot land.");
	  return;
     }

     if (!Planet(planet->listref)) {
	  Notify(player, "You cannot land on that!");
	  return;
     }

     if (distance(&planet->listref->pos, &object->pos) > DOCK_RANGE) {
	  FNotify(player, "A ship must be within %d units of the target"
		  " to land.", (int)DOCK_RANGE);
	  return;
     }

     sprintf(buff, "#%d", planet->listref->data_object);
     setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
     sprintf(buff, RANGEF " " RANGEF " " RANGEF, planet->listref->pos.x, 
	     planet->listref->pos.y, planet->listref->pos.z);
     setAttrByName(object->data_object, STATUS_STARTPOS, buff);

     evTrigger(object, EVENT_LANDED, "d", planet->listref);
     evTrigger(planet->listref, EVENT_LANDED_ON, "d", object);
     evBroadcast(object, planet->listref, "has landed on", EVENT_LANDED_OTHER);
    
	if (space_info[object->space].flags & SPACE_LOGGED)
	  log_space("%s (#%d) has landed on %s (#%d).",
		    object->name, object->data_object, planet->listref->name, 
		    planet->listref->data_object);

     objRemoveObject(object);

     return;

}

void navOrbit(TAG *object, dbref player, int target)
{
     CONTACT *planet;
     char buff[SMALL_BUF_SIZE];

     planet = snsFindContactByNumber(object, target);

     if (planet==NULL) {
	  Notify(player, "Invalid target.");
	  return;
     }

     if (!Planet(planet->listref)) {
	  Notify(player, "You cannot orbit that!");
	  return;
     }

     if (distance(&planet->listref->pos, &object->pos) > DOCK_RANGE) {
	  FNotify(player, "A ship must be within %d units of the planet"
		  " to orbit it.", (int)DOCK_RANGE);
	  return;
     }

     if (object->shipdata->warp_speed - object->shipdata->warp_accel > 0) {
	 Notify(player, "The ship is going too fast to make orbit.");
	 return;
     }

     object->shipdata->warp_set_speed = 0.0;
     object->shipdata->warp_speed = 0.0;

     sprintf(buff, "#%d", planet->listref->data_object);
     setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
     sprintf(buff, RANGEF " " RANGEF " " RANGEF, planet->listref->pos.x, 
	     planet->listref->pos.y, planet->listref->pos.z);
     setAttrByName(object->data_object, STATUS_STARTPOS, buff);
     damWriteDamage(object->data_object, object->shipdata);

     evTrigger(object, EVENT_ORBITING, "d", planet->listref);
     evTrigger(planet->listref, EVENT_ORBITED, "d", object);
     evBroadcast(object, planet->listref, "is orbiting", EVENT_ORBITING_OTHER);
    
	if (space_info[object->space].flags & SPACE_LOGGED)
	  log_space("%s (#%d) has entered orbit with %s (#%d).",
		    object->name, object->data_object, planet->listref->name, 
		    planet->listref->data_object);

     return;
}

void navLaunch(TAG *object, dbref player, dbref data)
{
     dbref planetdata;
     TAG *planet;	
     int space;

     space = atoi(getAttrByName(data, STATUS_SPACE));
     planetdata = getDbrefByName(data, STATUS_DOCKED_AT);

     if (!ValidObject(planetdata)) {
	  Notify(player, "The ship is not on a legal planet.  Please"
		 " contact an admin.");
	  return;
     }

     if ((planet = objFindObject(planetdata)) == NULL) {
	  Notify(player, "The planet is not active.  Unable to leave.");
	  return;
     }

     if (!Planet(planet)) {
	  Notify(player, "The ship is not on a planet.");
	  return;
     }

     if (space_info[space].flags & SPACE_LOCKED) {
	  Notify(player, "Space is temporarily locked.  Please try again "
		 "later.");
	  return;
     }

     objAddObject(data);

     if ((object=objFindObject(data)) == NULL) {
	  Notify(player, "Unable to activate ship.  Please contact an admin.");
	  return;
     }

     evTrigger(object, EVENT_LAUNCHED, "d", planet);
     evTrigger(planet, EVENT_LAUNCHED_FROM, "d", object);
     evBroadcast(object, planet, "launced from", EVENT_LAUNCHED_OTHER);
	
     return;
}

void navDock(TAG *object, dbref player, TAG *dock, int type)
{
     SHIP *dockee;
     char buff[SMALL_BUF_SIZE];
     int shield;

     if (!Ship(dock)) {
	  Notify(player, "You cannot dock with that!");
	  return;
     }

     dockee = dock->shipdata;

     if (dockee->door_status == DOORS_NONE)  {
	  FNotify(player, "%s does not have a docking bay.", 
		  type ? "Your ship" : "The target");
	  return;
     }

     if (dockee->door_size < object->size) {
	  Notify(player, type ? "The tractor target is too large"
		 " to dock with you." : "Your ship is too large to dock with "
		 "that.");
	  return;
     }

     if (dock->speed > 0.0 || object->speed > 0.0) {
	  Notify(player, "Both your ship and the target must disengage"
		 " warp before docking may commence.");
	  return;
     }

     shield = calcFacingShield(object->pos, dock);

     if (dock->shipdata->shield_status[shield] == SHLD_UP) {
	  FNotify(player, "%s facing shield must be lowered before "
		  "ship can enter dock.", type ? "Your" : "Dock's");
	  return;
     }

     if (distance(&dock->pos, &object->pos) > DOCK_RANGE) {
	  FNotify(player, "A ship must be within %d units of the target"
		  " to dock.", (int)DOCK_RANGE);
	  return;
     }

     if (dockee->door_status != DOORS_OPEN) {
	  FNotify(player, "%s docking bay doors are closed.",
		  type ? "The ship's" : "Target");
	  return;
     }

     sprintf(buff, "#%d", dock->data_object);
     setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
     sprintf(buff, RANGEF " " RANGEF " " RANGEF, dock->pos.x,
	     dock->pos.y, dock->pos.z);
     setAttrByName(object->data_object, STATUS_STARTPOS, buff);

     evTrigger(object, EVENT_DOCKED, "d", dock);
     evTrigger(dock, EVENT_DOCKED_WITH, "d", object);
     evBroadcast(object, dock, "has docked with", EVENT_DOCKED_OTHER);  

	if (space_info[object->space].flags & SPACE_LOGGED)
	  log_space("%s (#%d) has docked with %s (#%d).",
		    object->name, object->data_object, dock->name, 
		    dock->data_object);

     objRemoveObject(object);

     return;

}

void navUndock(TAG *object, dbref player, dbref data)
{
     dbref dockdata;
     TAG *dock;
     int space;

     space = atoi(getAttrByName(data, STATUS_SPACE));
     dockdata = getDbrefByName(data, STATUS_DOCKED_AT);

     if (!ValidObject(dockdata)) {
	  Notify(player, "The ship is not in a legal dock.  Please"
		 " contact an admin.");
	  return;
     }

     if ((dock = objFindObject(dockdata)) == NULL) {
	  Notify(player, "The dock is not active.  Unable to leave.");
	  return;
     }

     if (!Ship(dock)) {
	  Notify(player, "The ship isn't currently docked.");
	  return;
     }

     if (dock->shipdata->door_status != DOORS_OPEN) {
	  Notify(player, "The docking bay doors are closed.  Unable "
		 "to leave.");
	  return;
     }

     if (dock->shipdata->shield_status[dock->shipdata->door_side] == SHLD_UP) {

	  FNotify(player, "Dock's %s shield must be lowered before "
		  "ship can leave dock.", shdFullName(dock, dock->shipdata->door_side));

	  return;
     }

     if (space_info[space].flags & SPACE_LOCKED) {
	  Notify(player, "Space is temporarily locked.  Please try again "
		 "later.");
	  return;
     }

     objAddObject(data);

     if ((object=objFindObject(data)) == NULL) {
	  Notify(player, "Unable to activate ship.  Please contact an"
		 " admin.");
	  return;
     }

     evTrigger(object, EVENT_UNDOCKED, "d", dock);
     evTrigger(dock, EVENT_UNDOCKED_FROM, "d", object);
     evBroadcast(object, dock, "has undocked from", EVENT_UNDOCKED_OTHER);

     return;

}

#ifdef ENABLE_ODOMETER

void navSetOdometer(TAG *object, int dist)
{
     if (Ship(object))
	  object->shipdata->odometer = dist;
}

void navWriteOdometer(dbref data, SHIP *ship)
{
     char buff[SMALL_BUF_SIZE];

     sprintf(buff, "%d", ship->odometer);

     setAttrByName(data, STATUS_ODOMETER, buff);
}

#endif

#ifdef ENABLE_TIME_TRACKER

void navSetTimeActive(TAG *object, time_t t)
{
     if (Ship(object))
	  object->shipdata->time_active = t;
}

void navWriteTimeActive(dbref data, SHIP *ship)
{
     char buff[SMALL_BUF_SIZE];
     time_t t;

     t = time(NULL) - ship->time_start;

     sprintf(buff, "%d", (int) (ship->time_active + t));

     setAttrByName(data, STATUS_TIME_ACTIVE, buff);
}

#endif

void navRangeString(TAG *object, range_t range, char *dist, unsigned int width)
{
     int    scale;
     float  converted_range;
     SCALER *pFactor;
     char   *pStr;

     converted_range = range * object->range_factor;

     if (converted_range <= 1.0) {

	  pFactor = range_multipliers;

	  for (scale = 0; pFactor[scale].scale != 0; scale++) {

	       sprintf(dist, "%f", converted_range * pFactor[scale].scale);
	       dist[width-strlen(pFactor[scale].unit)] = '\0';
			
	       if (atof(dist) != 0.0) {
		    sprintf(dist, "%f", converted_range * pFactor[scale].scale);
		    sprintf(&dist[width-strlen(pFactor[scale].unit)], "%s",
			    pFactor[scale].unit);
		    return;
	       }
	  }

	  strncpy(dist, "0.00000000000000000", width);
	  dist[width] = '\0';

     } else {

	  pFactor = range_divisors;

	  for (scale = 0; pFactor[scale].scale != 0; scale++) {

	       sprintf(dist, "%f", converted_range / pFactor[scale].scale);
	       dist[width-strlen(pFactor[scale].unit)] = '\0';

	       /* If the '.' is in the string, we've not overflown the
                  buffer */
	       pStr = strchr(dist, '.');
	       if (pStr != NULL) {

				/* Zap the '.' if it's at the end */
		    if (pStr[1] == '\0')
			 *pStr = '\0';

		    strcat(dist,pFactor[scale].unit);

		    return;
	       }

	  }

	  strncpy(dist, ">999999999999999999", width);
	  dist[width] = '\0';
     }

}
