/* v0.9
 *
 * flag.c:  PSE flag maintenance routines.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "hashtab.h"
#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "flag.h"
#include "smisc.h"

/* Module internal structures */

typedef struct {
    const char  *name;   /* The name of the flag */
    unsigned int num;    /* The number of the flag. A mask */
    int          len;    /* The length of the name, avoid repeated work */
} SFLAGENT;

typedef struct {
    SFLAGENT    *fent;     /* Pointer to the SFLAGENT list */
    HNODE       *htab;     /* Pointer to the hash table    */
    int         num_flags; /* Number of flags in the list  */
    int         max_flags; /* Maximum number of flags in the list */
} SFLAGLIST;

/*
 * Declarations of the flag lists used in the PSE. Static as all of the
 * references to them are via the FLAGS enum index.
 */

static SFLAGENT objflaglist[] = {
    {"SHIP", TYPE_SHIP, 0},
    {"PLANET", TYPE_PLANET, 0},
    {"CAN_MOVE", CAN_MOVE, 0},
    {"MANUAL_POS", MANUAL_POS, 0},
    {"MANUAL_DIR", MANUAL_DIR, 0},
    {"EVENT_DRIVEN", EVENT_DRIVEN, 0},
    {"CAN_SENSE", CAN_SENSE, 0},
    {"CAN_BEAM_TO", CAN_BEAM_TO, 0},
    {"BEAMABLE", BEAMABLE, 0},
    {"ATTACKABLE", ATTACKABLE, 0},
    {"HUGE", HUGE_OBJ, 0},
    {"OMNISCIENT", OMNISCIENT, 0},
    {"NEARSIGHTED", NEARSIGHTED, 0},
    {"INVISIBLE", INVISIBLE, 0},
    {"HAZY", HAZY, 0},
    {"CATARACTS", CATARACTS, 0},
    {"CLOAKED", CLOAKED, 0},
    {"CAN_BEAM_FROM", CAN_BEAM_FROM, 0},
    {"REMOVED", REMOVED, 0},
    {"TRACTORABLE", TRACTORABLE, 0},
    {"", 0, 0}};

static SFLAGENT shipflaglist[] = {
    {"INVINCIBLE", INVINCIBLE, 0},
    {"SPEEDY", SPEEDY, 0},
    {"POKEY", POKEY, 0},
    {"NAKED", NAKED, 0},
    {"DISABLED", DISABLED, 0},
    {"PACIFIST", PACIFIST, 0},
    {"CAN_LAND", CAN_LAND, 0},
    {"", 0, 0}};

static SFLAGENT conflaglist[] = {
    {"navigation", CONS_NAV, 0},
    {"tactical", CONS_TAC, 0},
    {"engineering", CONS_ENG, 0},
    {"shields", CONS_SHD, 0},
    {"transporters", CONS_TRANS, 0},
    {"damage", CONS_DAM, 0},
    {"communications", CONS_COMM, 0},
    {"active", CONS_ACTIVE, 0},
    {"", 0, 0}};

/*
 * The index of flag lists used by this module.
 */
static SFLAGLIST flaglists[MAX_FLAG_LISTS] = {
    { objflaglist, NULL, 0},
    { shipflaglist, NULL, 0},
    { conflaglist, NULL, 0}};

/*
 * Used for plugin flag list allocations.
 */
static int flag_alloc;

/*
 * Called once as part of the PSE initialization code to build the flag list
 * hash tables.
 */
void flagInit()
{
    int ent, lst;
    SFLAGENT *sfe;

    for (lst = 0; lst < FLIST_END; lst++) {
	
	sfe = flaglists[lst].fent;
	flaglists[lst].htab = NULL;
	
	for (ent = 0; sfe[ent].name[0] != '\0'; ent++) {
	    sfe[ent].len = strlen(sfe[ent].name);
	    hashInsert(sfe[ent].name, (void *)(&sfe[ent]),
		       &flaglists[lst].htab);
	}
	
	flaglists[lst].num_flags = ent;
	flaglists[lst].max_flags = ent;
    }
    
    for (lst = FLIST_END; lst < MAX_FLAG_LISTS; lst ++) {
	flaglists[lst].fent = NULL;
	flaglists[lst].htab = NULL;
	flaglists[lst].num_flags = 0;
	flaglists[lst].max_flags = 0;
    }

    /* First unused entry */
    flag_alloc = FLIST_END;

    return;
}

/*
 * Code to read in a list of upto 32 (64 on 64bit machines) flags from a
 * binary list.
 *
 * Returns 0 if the string being processed does not consist solely of 0s and
 * 1s.
 */
int loadBinaryFlags(unsigned int *flags, const char *flaglist)
{
    int i;
    unsigned int flag, mask;
    
    flag = *flags = 0;
    
    for (i=0, mask=1; (flaglist[i] != '\0') && (mask != 0); i++, mask <<= 1) {
	
	if (flaglist[i] == '1')
	    flag += mask;
	else if (flaglist[i] != '0')
	    return 0;
    }
    
    *flags = flag;
    
    return 1;
}

/*
 * Code to write out a binary list of flags. This is being phased out in
 * favour of more human-friendly storage, and is only here for completeness.
 */
static void saveBinaryFlags(unsigned int flags, int bits, char *flaglist)
{
    int i;
    unsigned int mask;
    
    for (i = 0, mask = 1; i < bits; i++, mask <<=1)
	flaglist[i] = (flags & mask) ? '1' : '0';
    
    flaglist[bits] = '\0';
    
    return;
}

/*
 * A routine which parses up a provided flag list, and sets the provided
 * flags to the resultant value.  The function returns NULL if the string
 * was parsed successfully, and a pointer to the unrecognized flag if
 * not.
 */
const char *loadSymbolicFlags(unsigned int *flags, enum FLAGS flist,
			      const char *flagstr)
{
    const char *s;
    const char *t;
    int ret, len;
    
    s = flagstr;
    
    while(*s) {
	/* Skip leading whitespace */
	while(*s == ' ')
	    s++;
	
	/* find the end of the flag */
	t = s;
	len = 0;
	
	while(*t && *t != ' ') {
	    t++;
	    len++;
	}
	
	if (*s == '!')
	    ret = setFlagWithLen(flags, 0, flist, s+1, len-1);
	else
	    ret = setFlagWithLen(flags, 1, flist, s, len);
	
	/* If the lookup failed, return the pointer to the flag that failed */
	if (ret)
	    return s;
	
	/* Point to the start of the next word */
	s = t;
    }
    
    return NULL;
}

/*
 * A routine which parses up a provided flag list, and sets the provided
 * flags to the resultant value.  The function returns NULL if the string
 * was parsed successfully, and a pointer to the unrecognized flag if
 * not.
 */
const char *saveSymbolicFlags(unsigned int flags, enum FLAGS flist, int neg)
{
    static char buff[MAX_ATTRIBUTE_LEN];
    int i, len;
    const SFLAGENT *sfe;
    
    sfe = flaglists[flist].fent;
    
    for (i = len = 0; (*(sfe[i].name) != '\0') &&
	     (len+sfe[i].len+2 < MAX_ATTRIBUTE_LEN); i++) {
	
	if ((flags & sfe[i].num) != 0) {
	    
	    strcpy(&buff[len], sfe[i].name);
	    len += (sfe[i].len);
	    buff[len++] = ' ';
	    
	} else if (neg) {
	    
	    buff[len++] = '!';
	    strcpy(&buff[len], sfe[i].name);
	    len += (sfe[i].len);
	    buff[len++] = ' ';
	}
    }
    
    if (len > 0)
	len--;
    
    buff[len] = '\0';
    
    return buff;
}

/*
 * Search the provided flag list for the key name, and set (value != 0) or
 * clear (value = 0) the appropriate bit(s) in flags. The return value is
 * 1 if the key did not match, or 0 if is was matched.
 */
int setFlag(unsigned int *flags, int value, enum FLAGS flist,
	    const char *name)
{
    HNODE *htab = flaglists[flist].htab;
    SFLAGENT *flagptr;
    
    flagptr = (SFLAGENT *)hashSearch(name, htab);
    
    if (flagptr) {
	if (value)
	    *flags |= flagptr->num;
	else
	    *flags &= ~flagptr->num;
	
	return 0;
    }
    
    return 1;
}

/*
 * Search the provided flag list for the key name, and set (value != 0) or
 * clear (value = 0) the appropriate bit(s) in flags. The key length is
 * provided explicitly. The return value is 1 if the key did not match, or 0
 * if is was matched.
 */
int setFlagWithLen(unsigned int *flags, int value, enum FLAGS flist,
		   const char *name, int len)
{
    HNODE *htab = flaglists[flist].htab;
    SFLAGENT *flagptr;
    
    flagptr = (SFLAGENT *)hashSearchLen(name, htab, len);
    
    if (flagptr) {
	if (value)
	    *flags |= flagptr->num;
	else
	    *flags &= ~flagptr->num;
	
	return 0;
    }
    
    return 1;
}

/*
 * Search the specified flag list for the given flag. Returns 0 if the flag
 * is not set.
 */
int isFlagSet(unsigned int flags, enum FLAGS flist, const char *name)
{
    HNODE *htab = flaglists[flist].htab;
    SFLAGENT *flagptr;
    
    flagptr = (SFLAGENT *)hashSearch(name, htab);
    
    if (flagptr) {
	
	if ((flags & flagptr->num) != 0)
	    return 1;
    }
    
    return 0;
}

/*
 * Allocates a new flag list. The return number is the flag list id for this
 * list.
 */
int flagRegisterFlagList(int nflags)
{
    int lst;
    
    if (flag_alloc == MAX_FLAG_LISTS)
	return -1;
    
    if ((nflags > MAX_FLAGS_PER_LIST) || (nflags < 1))
	return -1;

    flaglists[flag_alloc].fent = pse_malloc(sizeof(SFLAGENT) * nflags);
    
    flaglists[flag_alloc].max_flags = nflags;
    
    lst = flag_alloc;
    flag_alloc++;
    
    return lst;
}

/*
 * Adds a flag to a flag list. The provided flag name is not copied.
 */
int flagRegisterFlagEntry(int lst, int bit, const char *name)
{
    SFLAGENT *sfe;
    
    /* Check for a valid flag list to add to */
    if ((lst >= flag_alloc) || (lst < FLIST_END))
	return -1;
    
    if ((bit < 0) || (bit >= MAX_FLAGS_PER_LIST))
	return -1;

    /* Check for a full flag list */
    if (flaglists[lst].num_flags == flaglists[lst].max_flags)
	return -1;
    
    sfe = &flaglists[lst].fent[flaglists[lst].num_flags];
    flaglists[lst].num_flags++;
    
    sfe->name = name;
    sfe->num = 1 << bit;
    sfe->len = strlen(name);
    hashInsert(sfe->name, (void *)sfe, &flaglists[lst].htab);
    
    return 1;
}
