/* v0.8
 *
 * tactical.c:  Ship tactical code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "tactical.h"
#include "sensors.h"
#include "damage.h"
#include "smisc.h"
#include "events.h"

typedef struct firing_entry FIREENT;

struct firing_entry {

	TAG *object;
	float offset;
	FIREENT *next;
};

FIREENT *fire_table = NULL;

void add_fire_table_entry(TAG *, TAG *, SPH);

void tacMaintainWeapons(TAG *object)
{
	SHIP *ship = object->shipdata;
	int i, j;
	int tacofficer = dbrefUser(ship->tactical);
	int weapon_charge;
	int num_ready = 0;
	int newly_finished = 0;

	/* charge gun banks that are online.  Just drain the others. */
	for (i = 0; i < ship->number_of_guns; i++) {
		ship->gun[i].old_charge = ship->gun[i].charge;
		ship->gun[i].charge *= 0.9;
		if (ship->gun[i].status == GUN_ONLINE) {
			weapon_charge = Min(ship->talloc_guns / 
			  ship->num_guns_online, 
			  ship->gun[i].charge_per_turn);

			ship->gun[i].charge = Min(ship->gun[i].charge + 
			  weapon_charge, ship->gun[i].power);

			if (ship->gun[i].charge == ship->gun[i].power) {
				num_ready = num_ready + 1;	

				/* check for a NEW full charge */
				if (ship->gun[i].old_charge != ship->gun[i].power) {

					/* just now fully charged.  Notify */
	  				FNotify(tacofficer, "%s %d fully charged.",
					  capstr(ship->gun_string), i);

					newly_finished = 1;
				}
			}
   		}
	}

	if (newly_finished && num_ready == ship->num_guns_online &&
	  EventDriven(object))
		evGunsCharged(object);

	newly_finished = 0;

	for (i = 0; i < ship->number_of_torps; i++) {

		if (ship->torp[i].status == TORP_ARMED ||
		  ship->torp[i].status == TORP_ONLINE) {
			/* increment and see if it is time to expire */
			if (++(ship->torp[i].turns_charged) > 
			  MAX_TURNS_CHARGED) {

				/* we have an expired torp.  zorch it */
				if (ship->torp[i].status == TORP_ONLINE)
			    		--(ship->num_torps_online);

				ship->torp[i].status = TORP_EMPTY;
				ship->torp[i].turns_charged = 0;
				ship->torp[i].charge = 0;

				FNotify(tacofficer, "%s %d expired.", 
				  capstr(ship->torp_string), i);

				/* Reload expired tube if possible */
	    			if (ship->free_torp_loaders &&
				  ship->auto_reload)
					tacReloadTorp(object, tacofficer, i);
			}
		}

		if (ship->torp[i].status == TORP_ONLINE) {
			/* calculate weapon charge.  Make sure it's legal 
			 * limit
			 */
			weapon_charge = Min(ship->talloc_torps / 
			  ship->num_torps_online, 
			  ship->torp[i].charge_per_turn);

			ship->torp[i].charge += weapon_charge;

			if (ship->torp[i].charge >= ship->torp[i].power/2) {

				ship->torp[i].status = TORP_ARMED;
				--(ship->num_torps_online);

				FNotify(tacofficer, "%s %d armed.", 
				  capstr(ship->torp_string), i);

				newly_finished = 1;
			}
		}

	}

	if (newly_finished && EventDriven(object) && ship->num_torps_online == 0
	  && ship->num_torp_loaders == ship->free_torp_loaders)
		evTorpsArmed(object);

	newly_finished = 0;

	for (i = 0; i < ship->number_of_torps; i++) {

		if (ship->torp[i].status == TORP_LOADING &&
			--(ship->torp[i].time_to_reload)==0) {
			ship->torp[i].status = TORP_LOADED;
			FNotify(tacofficer, "%s tube %d reloaded.", 
			  capstr(ship->torp_string), i);

			newly_finished = 1;
			
			/* if set for automatic charging, do it */
			if (ship->auto_online)
				tacTorpControl(object, tacofficer, i, 1);

			ship->free_torp_loaders++;

			if (ship->auto_reload) 
				for (j = 0; j < ship->number_of_torps; j++) 
					if (ship->torp[j].status == 
					  TORP_EMPTY) {
						tacReloadTorp(object, 
						  tacofficer, j);
						break;
					}
		}

	}

	if (newly_finished && EventDriven(object) &&
	  ship->num_torp_loaders == ship->free_torp_loaders)
		evTorpsLoaded(object);

	/* Tractor time */

	if ((ship->tractor_status == TRACTOR_ON || 
	  ship->tractor_status == TRACTOR_DRAW) && ship->tractor_power < 
	  distance(object->pos, ship->tractor_target->pos)) {
		Notify(dbrefUser(ship->tactical), "Tractor beam target has moved "
		  "out of range.");
		tacBreakTractor(object);
	}	

	return;
}

void tacAllocCheck(TAG *object)
{
	SHIP *ship = object->shipdata;
	dbref tacofficer;

	/* we don't know the tacofficer so find him */
	tacofficer = dbrefUser(ship->tactical);

	if (ship->talloc_guns + ship->talloc_torps + ship->talloc_tractor
	  > ship->alloc_tac) {
		Notify(tacofficer, "Energy allocation cut: ");
		tacAllocate(object, tacofficer, ship->talloc_guns,
		  ship->talloc_torps, ship->talloc_tractor);
	}
	else 
		FNotify(tacofficer, "Energy allocation cut.  New allocation is "
		  "%d.  No shortfall.", ship->alloc_tac);
	

	return;

}

void tacAllocate(TAG *object, dbref player, int guns, int torps, 
  int tractor)
{
	SHIP *ship = object->shipdata;
	int available_power;
	int total_allocation;
	float power_ratio;
	char buff[SMALL_BUF_SIZE];

	/* calculate max available power */
	available_power = ship->alloc_tac;

	/* set negative allocations to zero */
	if (guns < 0)
		guns = 0;
	if (torps < 0)
		torps = 0;

	if ((guns > 0 || torps > 0) && Pacifist(ship)) {
		Notify(player, "OOC:  Weapon use is temporarily disabled.");
		guns = 0;
		torps = 0;
	}
		

	/* calculate total allocation */
	total_allocation = guns + torps + tractor;

	/* if allocation > power, apply cuts */
	if (total_allocation > available_power) {
		power_ratio = (float)((float)available_power / 
		  (float)total_allocation);
		guns = (int)((float)guns * power_ratio);
		torps = (int)((float)torps * power_ratio);
		tractor = (int)((float)tractor * power_ratio);
	}

	if (object->shipdata->tractor_target != NULL &&
		Ship(object->shipdata->tractor_target)) {

		if (tractor > ship->talloc_tractor)
			Notify(dbrefUser(
			  object->shipdata->tractor_target->shipdata->tactical),
			  "Tractor beam source has increased the tractor "
			  "beam's power.");
		else if (tractor < ship->talloc_tractor)
			Notify(dbrefUser(
			  object->shipdata->tractor_target->shipdata->tactical),
			  "Tractor beam source has decreased the tractor "
			  "beam's power.");

	}

	/* store these allocations */
	ship->talloc_guns = guns;
	ship->talloc_torps = torps;
	ship->talloc_tractor = tractor;

	if (ship->tractor_status == TRACTOR_ON || ship->tractor_status == 
	  TRACTOR_DRAW)
		ship->tractor_power = (int)((float) ship->talloc_tractor * 
		  ship->tractor_effect / ship->tractor_target->size);

	FNotify(player, "Total available power: %d\nAllocations:",
	  available_power);

	sprintf(buff, "    %ss: %d  ", capstr(ship->gun_string), guns);

	FNotify(player,"%s %ss: %d  Tractor: %d", buff, 
	  capstr(ship->torp_string), torps, tractor);

}

void tacReloadTorp(TAG *object, dbref player, int tube)
{
	SHIP *ship = object->shipdata;

	/* Inactive check already done for tactical commands */

	if (tube < 0 || tube >= ship->number_of_torps) {
		Notify(player, "Invalid tube number.");
		return;
	}

	if (ship->torp[tube].status != TORP_EMPTY) {
		FNotify(player, "%s tube not empty.", 
		  capstr(ship->torp_string));
		return;
	}

	if (ship->torp[tube].status == TORP_LOADING) {
		FNotify(player, "We are already loading tube %d", tube);
		return;
	}

	if (ship->torp_ammo == 0) {
		FNotify(player, "The torpedo supply is empty.");

		if (ship->auto_reload == 1) {
			FNotify(player, "%s auto-reloading disabled.",
			  capstr(ship->torp_string));
			ship->auto_reload = 0;
		}

		return;
	}

	if (ship->free_torp_loaders <= 0) {
		FNotify(player, "All of the %s loaders are in use.", 
		  ship->torp_string);  
		return;
	}

	ship->torp[tube].status = TORP_LOADING;
	ship->torp[tube].time_to_reload = TURNS_TO_LOAD;
	ship->free_torp_loaders--;

	if (ship->torp_ammo > 0)
		ship->torp_ammo--;

	FNotify(player, "Reloading tube %d.", tube);
	return;

}

void tacGunControl(TAG *object, dbref player, int bank, int action)
{
	SHIP *ship = object->shipdata;

	/* valid bank check */
	if (bank < 0 || bank >= ship->number_of_guns) {
		FNotify(player, "Invalid %s number.", ship->gun_string);
		return;
	}

	switch (ship->gun[bank].status) {
	  case GUN_OFFLINE:
		if (action) {
			FNotify(player, "%s %d set online.", 
			  capstr(ship->gun_string), bank);
			ship->gun[bank].status = GUN_ONLINE;
			++(ship->num_guns_online);
		}
		else
			FNotify(player, "%s %d already offline.", 
			  capstr(ship->gun_string), bank);
		break;
	  case GUN_ONLINE:
		if (!action) {
			FNotify(player, "%s %d set offline.",
			  capstr(ship->gun_string), bank);
			  ship->gun[bank].status = GUN_OFFLINE;
			  --(ship->num_guns_online);
		}
		else
			FNotify(player, "%s %d already on line.", 
			  capstr(ship->gun_string), bank);
		break;
	  case GUN_DAMAGED:
		FNotify(player, "%s %d damaged.", capstr(ship->gun_string), 
		  bank);
		break;
	}

	return;
}

void tacTorpControl(TAG *object, dbref player, int tube, int action)
{
	SHIP *ship = object->shipdata;

	/* valid tube check */
	if (tube < 0 || tube >= ship->number_of_torps) {
		Notify(player, "Invalid tube number.");
		return;
	}

	if (!action) {
		if (ship->torp[tube].status == TORP_ONLINE) {
			FNotify(player, "%s %d taken offline.",
				capstr(ship->torp_string), tube);
			ship->torp[tube].status = TORP_LOADED;
			ship->torp[tube].charge = 0;
			ship->torp[tube].turns_charged = 0;
		}
		else
			FNotify(player, "%s not online.", 
			  capstr(ship->torp_string));

		return;
	}
	
	switch (ship->torp[tube].status) {
	  case TORP_EMPTY:
		FNotify(player, "%s must be loaded before it can be armed.",
		  capstr(ship->torp_string));
		break;
	  case TORP_LOADED:
		Notify(player, "Charging.");
		++(ship->num_torps_online);
		ship->torp[tube].status = TORP_ONLINE;
		break;
	  case TORP_ONLINE:
		Notify(player, "Already charging.");
		break;
	  case TORP_ARMED:
		Notify(player, "Already armed.");
		break;
	  case TORP_DAMAGED:
		FNotify(player, "%s damaged and cannot be charged.",
		  capstr(ship->torp_string));
		break;
	}

	return;

}

void tacBreakLock(TAG *object)
{
	dbref tacofficer;
	CONTACT *target = NULL;

	/* get the tacofficer */
	if (Ship(object->locked_on->listref))
		tacofficer = dbrefUser(object->locked_on->listref->shipdata->tactical);
	else
		tacofficer = -1;

	/* Check the target's contact list for the breaker. */
	target = snsFindContact(object->locked_on->listref, object);

	/* Okay.  Notify the other guy now, but only if he's alive. */
	if (tacofficer > 0) {
		if (target == NULL) 
			Notify(tacofficer, "Weapons lock from unknown "
			  "source has been broken.");
		else 
			FNotify(tacofficer, "Weapons lock from contact "
			  "[%d] %s has been broken.", 
			  target->contact_number, target->listref->name);

	}

	if (EventDriven(object->locked_on->listref))
		evLockBroken(object, object->locked_on->listref);

	/* break the lock */
	object->locked_on = NULL;
	return;
}

void tacLockWeapons(TAG *object, dbref player, int num)
{
	CONTACT *target, *us;

	target = snsFindContactByNumber(object, num);

	if (target==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (!Attackable(target->listref)) {
		Notify(player, "You cannot lock weapons on that target.");
		return;
	}

	if (object->locked_on != NULL)  {
		if (object->locked_on == target) {
			Notify(player, "We are already locked on that target.");
			return;
		}
		else
			tacBreakLock(object);
	}

	object->pending_lock = target;

	/* Give 'em an out-of-turn chance to spot us. */
	snsSenseObject(target->listref, object);

	if (Ship(target->listref)) {
		us = snsFindContact(target->listref, object);

		if (us != NULL)
			FNotify(dbrefUser(target->listref->shipdata->tactical),
			  "Warning!  Contact [%d]: %s is locking weapons.", 
			  us->contact_number, object->name);
		else
			FNotify(dbrefUser(target->listref->shipdata->tactical),
			  "Warning!  We are being locked on by an unknown"
			  " source.");
	}

	Notify(player, "Attempting to lock weapons.");

	if (EventDriven(target->listref))
		evLockedOn(target->listref, object);

	return;

}

void tacUnlock(TAG *object, dbref player)
{

	if (object->pending_lock != NULL) {
		Notify(player, "Lock attempt aborted.");
		object->pending_lock = NULL;
		return;
	}

	if (object->locked_on == NULL) 
		Notify(player, "Weapons are currently not locked.");

	else {
		tacBreakLock(object);
		Notify(player, "Weapons unlocked.");
	}

	return;

}	

/* tacFireGun:  Adapted from the original Chuckles.  Please send me 
 * suggestions for improvements.  I think it should be possible to miss
 * small/fast ships.  
 *
 * Return -1 for guns which don't fire.
 */
int tacFireGun(TAG *object, dbref player, int bank)
{
	SHIP *ship = object->shipdata;
	CONTACT *contact = object->locked_on;
	float effective_max_range;
	float range;
	int damage;
	float damage_coeff;

	/* If we're not locked on, we don't fire */
	if (contact == NULL)
		return -1;

	/* compute the effective maximum range */
	effective_max_range = ship->gun[bank].range * 
	  ship->gun[bank].charge / ship->gun[bank].power;

	/* If the target is too far away, we don't fire */
	range = distance(object->pos, contact->listref->pos);
	if (range > effective_max_range) {
		FNotify(player, "%s %i's max range at current charge is %1.0f.",
			capstr(ship->gun_string), bank, effective_max_range);
		return -1;
	}

	/* Okay. We fired. */
	damage_coeff = ship->gun[bank].range_percent + 
	  (FRAND - 0.5) * 0.5 * ship->gun[bank].range_percent;

	damage = (int) ((float)(ship->gun[bank].charge) *
	  (1.0 + (damage_coeff - 1.0) * range / effective_max_range) + 0.5);

	FNotify(player, "%s %d fired.  Hit for %2.0f%% yield.",
	  capstr(ship->gun_string), bank, 
	  (float)damage/(float)(ship->gun[bank].charge) * 100.0);

	/* Discharge the weapon */
	ship->gun[bank].charge = 0;

	/* Flag the amount of damage this hit did ( >= 0 ) */
	return damage;
}

void tacFreeFireTable(void)
{
	FIREENT *ftmp = NULL, *fptr;

	fptr = fire_table;

	while (fptr != NULL) {
		ftmp = fptr;
		fptr = fptr->next;

		free(ftmp);
	}

	fire_table = NULL;
	return;
}	

void tacInitFireTable(TAG *object, SPH direction)
{
	DISTENT *ptr, *entry;
	int range;

	/* This is a bit messy, but we don't store this value anywhere. */
	range = atoi(getAttrByName(object->data_object, CONFIG_TORP_RANGE));

	for (entry=dist_table[object->space];entry != NULL;entry=entry->next) {
		if (entry->object == object)
			break;
	}

	if (entry==NULL)
		return;

	for (ptr=entry->next; ptr != NULL; ptr=ptr->next) {
		if (ptr->range - entry->range < range) {
			if (Attackable(ptr->object))
				add_fire_table_entry(object, ptr->object, 
				  direction);
		}
		else
			break;
	}

	for (ptr=entry->prev; ptr != NULL; ptr=ptr->prev) {
		if (entry->range - ptr->range < range)	
			add_fire_table_entry(object, ptr->object, direction);
		else
			break;
	}
	
	return;
}
	
void add_fire_table_entry(TAG *object, TAG *target, SPH direction)
{
	float bear_diff;
	float elev_diff;
	XYZ rel_pos;
	SPH bearing;
	FIREENT *new_entry, *fprev=NULL, *fptr;

	rel_pos.x = target->pos.x - object->pos.x;
	rel_pos.y = target->pos.y - object->pos.y;
	rel_pos.z = target->pos.z - object->pos.z;

	xyz_to_sph(rel_pos, &bearing);

	bear_diff = bearing.bearing - direction.bearing;
	elev_diff = bearing.elevation - direction.elevation;

	new_entry = (FIREENT *)malloc(sizeof(FIREENT));	
	new_entry->object = target;
	new_entry->offset = sqrt(bear_diff*bear_diff + elev_diff*elev_diff);
	new_entry->next = NULL;

	for (fptr=fire_table; fptr != NULL; fptr=fptr->next) {

		/* 
		 * Make sure locked targets move closer to the front of
		 * the list.
		 */
		if (object->locked_on && new_entry->offset == fptr->offset &&
		  object->locked_on->listref == target)	
			break;

		if (new_entry->offset < fptr->offset)
			break;

		fprev = fptr;
	}		

	if (fprev == NULL) {
		new_entry->next = fire_table;
		fire_table = new_entry;
	}
	else {
		new_entry->next = fprev->next;
		fprev->next = new_entry;
	}

	return;
}

void tacRemoveFireTableEntry(TAG *object)
{

	FIREENT *ptr, *prev;

	if (fire_table == NULL)
		return;

	prev = NULL;

	for (ptr = fire_table; ptr != NULL; ptr=ptr->next) {

		if (ptr->object == object) {

			if (prev == NULL)
				fire_table = ptr->next;
			else
				prev->next = ptr->next;

			free(ptr);

			break;
		}

		prev = ptr;

	}				

	return;

}

/* tacFireTorp:  Fire torpedo.  Here's the basic explanation.  A list
 * is first created, the "firing list", containing the locked target (if any)
 * followed by nearby ships.  If there is no lock, the list contains ships
 * ordered by their angular distance from the firing direction.  A hit roll
 * is made on each ship in the list, in order, until a ship is hit, or the
 * list is exhausted.  Penalties apply if there is no lock, and more severe
 * penalties apply if the shot is stray -- if we are checking the to hit roll
 * on a nearby ship if the locked on ship has been missed.  So, firing into
 * a mix of friendly/enemy ships is realistically not a good idea.  This can 
 * make it possible to randomly try to hit cloakers though.  
 */
int tacFireTorp(TAG *object, dbref player, int tube, SPH direction, int nolock)
{

	SHIP *ship = object->shipdata;
	CONTACT *contact;
	float prob;
	int hit;
	FIREENT *ptr;
	TAG *target;
	int dist;
	XYZ loc;

	/* fire the torps */

	for (ptr = fire_table; ptr != NULL; ptr = ptr->next) {

		dist = distance(object->pos, ptr->object->pos);

		if (direction.range > 0 && direction.range < dist)
			continue;

		prob = (1.0 - (dist / (ship->torp[tube].range * 
		  ptr->object->size)));

		if (nolock)
			prob *= 0.5;	/* penalty for no lock */
		else if (object->locked_on->listref != ptr->object)
			prob *= 0.25;	/* penalty for stray shot */
		else
			prob *= ship->torp[tube].base_accuracy;

		prob = prob / (ptr->offset/2.0 + 1.0);

		if (FRAND < prob)
			break;
	}


	if (ptr != NULL) {
		target = ptr->object;

		if (!nolock && object->locked_on->listref == target) {
			FNotify(player, "%s %d hit!", 
			  capstr(ship->torp_string), tube);
			hit = 1;
		}
		else {
			/* Stray hit!/Hit with no lock! */

			contact = snsFindContact(object, target);

			if (contact != NULL)
				FNotify(player, "%s %d hit %s [%d]!", 
				  capstr(ship->torp_string), tube,
				  contact->listref->name, 
				  contact->contact_number);
			else
				FNotify(player, "%s %d hit an unknown target!", 
				  capstr(ship->torp_string), tube);

			if (Ship(target)) {
				contact = snsFindContact(target, object);
				if (contact != NULL)
				  FNotify(dbrefUser(target->shipdata->tactical),
				    "%s %s from %s [%d] hit!",
				    nolock ? "Unlocked" : "Stray",
				    target->shipdata->torp_string,
				    object->name, contact->contact_number);
				else
				  FNotify(dbrefUser(target->shipdata->tactical),
				    "%s hit from an unknown source!",
				    target->shipdata->torp_string);
			}

			evAttacked(target, contact);

			if (object->space == 0)
				log_space("%s (#%d) hit %s (#%d) with %s torp.",
				  object->name, object->data_object, 
				  target->name, target->data_object,
				  nolock ? "an unlocked" : "a stray");

			damBattleDamage(object->pos, target, 
			  ship->torp[tube].power, WPN_TORP);

			hit = 0;
		}

	}	
	else {

		if (direction.range > 0) {

			if (direction.range > ship->torp[tube].range)
				FNotify(player, "%s %d failed to detonate. ",
				  capstr(ship->torp_string), tube);
			else {
				FNotify(player, "Detonating %s %d at range %d.",
				  ship->torp_string, tube, direction.range);

				/* Detonate the torp */
				sph_to_xyz(direction, &loc);
	
				loc.x += object->pos.x;
				loc.y += object->pos.y;
				loc.z += object->pos.z;

				log_space("%s (#%d) detonated a torp.",
				  object->name, object->data_object);
	
				damExplosionDamage(object->space, loc, 
				  ship->torp[tube].power / 2);
			}

		}
		else	
			FNotify(player, "%s %d missed.", 
			  capstr(ship->torp_string), tube);

		hit = 0;
	}

	ship->torp[tube].status = TORP_EMPTY;
	ship->torp[tube].charge = 0;
	ship->torp[tube].turns_charged = 0;

	if (ship->free_torp_loaders && ship->auto_reload)
		tacReloadTorp(object, player, tube);

	return hit;

}

void tacTractorLock(TAG *object, dbref player, int num)
{
	SHIP *ship = object->shipdata;
	CONTACT *target, *us;
	int power;

	if (ship->tractor_status == TRACTOR_NONE) {
		Notify(player, "This ship is not equipped with a tractor "
		  "beam generator.");
		return;
	}

	if (ship->tractor_status == TRACTOR_DAMAGED) {
		Notify(player, "The tractor beam generator is damaged.");
		return;
	}

	target = snsFindContactByNumber(object, num);

	if (target==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (Huge(target->listref)) {
		Notify(player, "You can't lock the tractor beam on that.");
		return;
	}

	power = (int)((float) ship->talloc_tractor * ship->tractor_effect / 
	  target->listref->size);

	if (power < distance(object->pos, target->listref->pos)) {
		Notify(player, "Target is too far away for the tractor beam to"
		  " be effective.");
		return;
	}

	if (ship->tractor_status == TRACTOR_ON) {
		if (ship->tractor_target == target->listref)  {
			Notify(player, "We are already locked on that target.");
			return;
		}
		else 
			tacBreakTractor(object);
	}

	ship->tractor_status = TRACTOR_ON;
	ship->tractor_target = target->listref;
	target->listref->tractor_source = object;
	ship->tractor_power = power;

	/* Give the opponent a chance to sense the engager */
	snsSenseObject(target->listref, object);

	if (Ship(target->listref)) {
		us = snsFindContact(target->listref, object);

		if (us != NULL)
			FNotify(dbrefUser(target->listref->shipdata->tactical),
			  "Warning!  Contact [%d]: %s has locked on a tractor"
			  " beam.", us->contact_number, object->name);
		else
			FNotify(dbrefUser(target->listref->shipdata->tactical),
			  "Warning!  A tractor beam has been locked on us "
			  "by an unknown source.");
	}
	
	Notify(player, "Engaging tractor beam.");

	if (EventDriven(target->listref))
		evTractorOn(target->listref, object);

}

void tacTractorUnlock(TAG *object, dbref player)
{
	SHIP *ship = object->shipdata;

	if (ship->tractor_status == TRACTOR_OFF) {
		Notify(player, "The tractor beam is not presently engaged.");
		return;
	}
	else {
		Notify(player, "Tractor beam disengaged.");
		tacBreakTractor(object);
	}

	return;
}

void tacBreakTractor(TAG *object)
{
	SHIP *ship = object->shipdata;
	dbref tacofficer;
	CONTACT *target;

	/* get the tacofficer */
	if (Ship(ship->tractor_target))
		tacofficer = dbrefUser(ship->tractor_target->shipdata->tactical);
	else
		tacofficer = -1;

	/* Check the target's contact list for the breaker. */
	target = snsFindContact(ship->tractor_target, object);

	if (tacofficer > 0) {

		if (target == NULL) 
			Notify(tacofficer, "Tractor beam from unknown "
			  "source has been broken.");
		else 
			FNotify(tacofficer, "Tractor beam from contact "
			  "[%d] %s has been broken.", 
			  target->contact_number, target->listref->name);

	}

	if (EventDriven(ship->tractor_target))
		evTractorBroken(object, ship->tractor_target);

	ship->tractor_status = TRACTOR_OFF;
	ship->tractor_target->tractor_source = NULL;
	ship->tractor_target = NULL;
	ship->tractor_power = 0;

	return;
}	
