/* v0.8
 *
 * smisc.c:  Miscellaneous code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "smisc.h"

void FNotify(dbref target, char *message, ...)
{
	char buff[LARGE_BUF_SIZE];
	va_list list;

	va_start(list, message);
	vsprintf(buff, message, list);
	va_end(list);

	Notify(target, buff);
} 

void sph_to_xyz(SPH coord, XYZ *new_coord)
{
	float r = (float) coord.range + 0.9;

	new_coord->x = r * (cos(coord.bearing * PI / 180.0) * 
	  cos(coord.elevation * PI / 180.0));
	new_coord->y = r * (sin(coord.bearing * PI / 180.0) * 
	  cos(coord.elevation * PI / 180.0));
	new_coord->z = r * (sin(coord.elevation * PI / 180.0));

	return;
}

void xyz_to_sph(XYZ coord, SPH *new_coord)
{
	double x = (double) coord.x;
	double y = (double) coord.y;
	double z = (double) coord.z;

	if (y == 0)
	{
		if (x >= 0) new_coord->bearing = 0.0;
		else new_coord->bearing = 180.0;
	}
	else if (x == 0)
	{
		if (y > 0.0) new_coord->bearing = 90.0;
		else new_coord->bearing = 270.0;
	}
	else
	{
		new_coord->bearing = atan(y / x) * 180.0 / PI;
		if (x < 0.0) new_coord->bearing +=180.0;
	}

	if (z == 0)
		new_coord->elevation = 0.0;
	else if (x == 0 && y == 0) {
		if (z > 0) new_coord->elevation = 90.0;
		else new_coord->elevation = -90.0;
	}
	else 
		new_coord->elevation = atan(z / sqrt((double) x*x +  
		  (double) y*y)) * 180.0 / PI;

	new_coord->range = (int) sqrt((double) x*x + (double) y*y + (double) z*z);

	/* bring bearing into range */
	if (new_coord->bearing < 0.0) new_coord->bearing += 360.0;
	if (new_coord->bearing >= 360.0) new_coord->bearing -=360.0;

	return;
}

int distance(XYZ a, XYZ b)
{
	double dx, dy, dz;
	double sum;

	dx = (double) b.x - a.x;
	dy = (double) b.y - a.y;
	dz = (double) b.z - a.z;

	sum = dx*dx + dy*dy + dz*dz;
	
	return (sqrt(sum));
}

void log_space(char *message, ...)
{
	char buff[LARGE_BUF_SIZE];
	va_list list;

	va_start(list, message);
	vsprintf(buff, message, list);
	va_end(list);
	fprintf(stderr, "SPACE- %s\n", buff);

	if (getDebugCharacter() >= 0)
		FNotify(getDebugCharacter(), "SPACE- %s", buff);
}

void fillXYZCoords(dbref obj, char *attrname, XYZ *coords)
{
	char *s, *ptr;

	s=getAttrByName(obj, attrname);

	if (!strlen(s)) return;

	ptr=strtok(s, " ");
	if (ptr) coords->x = atoi(ptr);
	ptr=strtok(NULL, " ");
	if (ptr) coords->y = atoi(ptr);
	ptr=strtok(NULL, " ");
	if (ptr) coords->z = atoi(ptr);

	return;
}

void fillSPHCoords(dbref obj, char *attrname, SPH *coords)
{
	char *s, *ptr;

	s=getAttrByName(obj, attrname);

	if (!strlen(s)) return;

	ptr=strtok(s, " ");
	if (ptr) coords->bearing = atof(ptr);
	ptr=strtok(NULL, " ");
	if (ptr) coords->elevation = atof(ptr);
	ptr=strtok(NULL, " ");
	if (ptr) coords->range = atoi(ptr);

	return;
}

int parseDbref(char *s)
{

	char *p;
	int x;

	if (strlen(s) > 1) {
		for (p = s+1; *p; p++) {
			if (!isdigit(*p))
				return NOTHING;
		}
	}
	else
		return NOTHING;

	x = atoi(s+1);
	return ((x >= 0) ? x : NOTHING);
}

int calcFacingShield(XYZ source, TAG *target)
{
	float dx1, dy1, dz1, dx2, dy2, dz2, dx3, dy3, dz3;
	float theta, psi;
	float angle_from_front;

	dx1 = (float) source.x - target->pos.x;
	dy1 = (float) source.y - target->pos.y;
	dz1 = (float) source.z - target->pos.z;

	psi = (target->heading.bearing * PI / 180.0);
	theta = (target->heading.elevation * PI / 180.0);

	/* first transformation -- NOTICE Euler is using Euler angles.
	 * If you don't believe it, look them up.  How ironic.
	 */
	dx2 = dx1*cos(psi) + dy1*sin(psi);
	dy2 = dy1*cos(psi) - dx1*sin(psi);
	dz2 = dz1;

	/* second transformation */
	dx3 = dx2*cos(theta) + dz2*sin(theta);
	dy3 = dy2;
	dz3 = dz2*cos(theta) - dx2*sin(theta);

	if (dx3 == 0.0)
		angle_from_front = 90.0;
	else
		angle_from_front = acos(dx3 / sqrt((double) dx3*dx3 + (double) dy3*dy3 +
		  (double) dz3*dz3)) * 180.0 / PI;

	/* return the appropriate facing shield */   

	if (angle_from_front <= 60.0) return FORE_SHIELD;
	if (angle_from_front >= 120.0) return AFT_SHIELD;
	if (dy3 < 0.0) return PORT_SHIELD;
	return STARBOARD_SHIELD;

}

char *capstr(char *string)
{
	static char str[100];

	sprintf(str, "%c%s", toupper(string[0]), string+1);

	return str;
}
