/* v0.8
 *
 * shields.c:  Ship shield code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "damage.h"
#include "shields.h"
#include "smisc.h"
#include "events.h"

void shdAllocCheck(TAG *object)
{
	SHIP *ship = object->shipdata;
	int i;
	int allocation = 0;
	dbref shdofficer;

	shdofficer = dbrefUser(ship->shields);

	for (i=0; i < 4; i++) 	
		allocation += ship->shield_alloc[i];	

	if (allocation > ship->alloc_shields) {
		Notify(shdofficer, "Energy allocation cut:");
		shdAllocate(object, shdofficer, 
		  ship->shield_alloc[0], ship->shield_alloc[1], 
		  ship->shield_alloc[2], ship->shield_alloc[3]);
	}
	else 
		FNotify(shdofficer, "Energy allocation cut.  New allocation is "
		  "%d.  No shortfall.", ship->alloc_shields);
	
	return;
}


void shdAllocate(TAG *object, dbref player, int fore, int aft, int port, 
  int starboard)
{
	SHIP *ship = object->shipdata;
	int total_allocation;
	float power_ratio;

	/* set any negative allocations to zero */
	if (fore < 0) fore = 0;
	if (aft < 0) aft = 0;
	if (port < 0) port = 0;
	if (starboard < 0) starboard = 0;

	total_allocation = fore + aft + port + starboard;

	/* check for overallocation */
	if (total_allocation > ship->alloc_shields) {
		power_ratio = (float)(ship->alloc_shields) / 
		  (float)total_allocation;
		fore = (int)((float)fore * power_ratio);
		aft = (int)((float)aft * power_ratio);
		port = (int)((float)port * power_ratio);
		starboard = (int)((float)starboard * power_ratio);
	}

	/* assign the new values */
	ship->shield_alloc[0] = fore;
	ship->shield_alloc[1] = aft;
	ship->shield_alloc[2] = port;
	ship->shield_alloc[3] = starboard;

	/* check for cloak power */
	if (ship->cloak_status == CLOAK_ON || ship->cloak_status == CLOAKING) {
		if ((ship->alloc_shields - total_allocation) < ship->cloak_cost) {
			if (ship->cloak_status == CLOAKING)
				Notify(player, "Insufficient energy to "
				  "continue establishing cloak.");
			else	
				Notify(player, "Insufficient energy to "
				  "maintain cloak.");

			FNotify(player, "Cloak requires %d unallocated power to shields.",
			  ship->cloak_cost);
			shdCloakOff(object, player);
		}
	}

	/* notify the player */
	FNotify(player, "Total available power: %d\nAllocations:", 
	  ship->alloc_nav);
	FNotify(player, "Shields:  fore:%4d  aft:%4d  port:%4d"
	  "  starboard:%4d", fore, aft, port, starboard);

}

void shdMaintainShields(TAG *object)
{

	SHIP *ship = object->shipdata;
	int shield_additions[4], old_shield_level[4], i;

	/* Store previous values, remove overload energy, and bleed */
	for (i = 0; i < 4; i++) {
		old_shield_level[i] = ship->shield_level[i];

		if (ship->shield_level[i] > ship->shield_max[i])
			ship->shield_level[i] = ship->shield_max[i];

		ship->shield_level[i] *= (ship->shield_status[i]==SHLD_READY) 
		  ?  SHLD_READY_BLEED : SHLD_UP_BLEED;

		/* Calculate shield additions */	
		shield_additions[i] = Min(ship->shield_alloc[i],
		  ship->shield_max_charge[i]);

      		/* now add any allocated points if the shield is working and      		 * truncate overloads at overload max and update action 
		 * registers
		 */

		if (ship->shield_status[i] != SHLD_DAMAGED)
			ship->shield_level[i] += (int)(shield_additions[i] *
			  ship->shield_factor);

		if ((Cloaked(object) || ship->cloak_status == CLOAKING) &&
		  ship->shield_level[i] > old_shield_level[i])
			ship->shield_level[i] = old_shield_level[i];

		if (ship->shield_status[i] != SHLD_UP)
			if (ship->shield_level[i] > ship->shield_max[i]) {
				ship->shield_level[i] = ship->shield_max[i];
				ship->shield_action[i] = SHLD_OVERLOADED;
			}
			else if (ship->shield_level[i] == old_shield_level[i])
				ship->shield_action[i] = SHLD_STABLE;
		else
			ship->shield_action[i] = (ship->shield_level[i] >
			  old_shield_level[i]) ? SHLD_CHARGING : SHLD_FALLING;
	}

	return;

}

void shdCloakOn(TAG *object, dbref player)
{
	SHIP *ship = object->shipdata;
	int i, total_allocation = 0;

	if (Naked(ship)) {
		Notify(player, "Cloak inoperable.");
		return;
	}

	/* make sure we're even allowed to cloak */
	switch (ship->damage[SYSTEM_CLOAK].status) {
	  case 'x':
	  case 'X':
		Notify(player, "This ship has no cloaking device.");
		return;
	  case '7':
	  case '9':
		Notify(player, "The cloaking device is damaged and may not be "
		  "used until repaired.");
		return;
	}

	/* make sure it isn't already on */
	if (Cloaked(object) || ship->cloak_status==CLOAKING) {
		Notify(player, "Cloak is already engaged.");
		return;
	}

	/* check for shields */
	for (i = 0; i < 4; i++) {
		if (ship->shield_status[i] == SHLD_DAMAGED) {
			Notify(player, "All shields must be undamaged to use "
			  "cloak.");
			return;
		}
		total_allocation += ship->shield_alloc[i];
	}

	/* Make sure the shield allocation is enough to run the cloak */

	if (ship->alloc_shields - total_allocation < ship->cloak_cost) {	
		Notify(player, "Insufficient energy to engage cloak.");
		FNotify(player, "Cloak requires %d unallocated power to shields.",
		  ship->cloak_cost);
		return;
	}

	/* Time to cloak.  Set the cloak status first so that the shield  *
	 * functions will know to suppress the messages.                  */

	ship->cloak_status = CLOAKING;
	ship->time_to_cloak = 2;

	for (i = 0; i < 4; i++) 
		shdLowerShield(object, player, i);

	Notify(player, "Engaging cloaking device.");

	/* notify anyone who can see us */
	evCloakEngaged(object);

	return;
}

void shdCloakOff(TAG *object, dbref player)
{
	SHIP *ship = object->shipdata;

	/* switch on the status */
	switch(ship->damage[SYSTEM_CLOAK].status) {
	  case 'x':
	  case 'X':
		Notify(player, "This ship has no cloaking device.");
		return;
	  case '7':
	  case '9':
		Notify(player, "The cloaking device is damaged and need"
		  " not be disengaged.");
		return;
	}

	/* terrific.  We're cloaked.  Change the status register */
	ship->cloak_status = DECLOAKING;
	object->tagflags &= ~CLOAKED;
	ship->time_to_cloak = 2;

	/* notify anyone who can see us */
	evCloakDisengaged(object);

	Notify(player, "Disengaging cloaking device.");
	return;
}

void shdRaiseShield(TAG *object, dbref player, int shield)
{
	SHIP *ship = object->shipdata;
	const char *shields[] = {"Fore", "Aft", "Port", "Starboard"};

	if (Naked(ship)) {
		Notify(player, "Shields inoperable.");
		return;
	}

	/* don't raise if cloaked */
	if (Cloaked(object) || ship->cloak_status == CLOAKING) {
		Notify(player, "Shields cannot be raised while "
		  "cloaking device is engaged.");
		return;
	}

	switch (ship->shield_status[shield]) {
	  case SHLD_READY:
		ship->shield_status[shield] = SHLD_UP;
		FNotify(player, "%s shield raised.",
		  shields[shield]);
		break;
	  case SHLD_UP:
		FNotify(player, "%s shield is already up.",
		  shields[shield]);
		break;
	  case SHLD_DAMAGED:
		FNotify(player, "%s shield is damaged and "
		  "may not be raised or lowered.", shields[shield]);
		break;
	}

	return;
}

void shdLowerShield(TAG *object, dbref player, int shield)
{
	SHIP *ship = object->shipdata;
	const char *shields[] = {"Fore", "Aft", "Port", "Starboard"};

	if (Naked(ship)) {
		Notify(player, "Shields inoperable.");
		return;
	}

	switch (ship->shield_status[shield]) {
	  case SHLD_READY:
		if (!Cloaked(object) && ship->cloak_status != CLOAKING)
			FNotify(player, "%s shield is already down.",
			  shields[shield]);
		break;
	  case SHLD_UP:
		ship->shield_status[shield] = SHLD_READY;
		if (!Cloaked(object) && ship->cloak_status != CLOAKING)
			FNotify(player, "%s shield lowered.",
			  shields[shield]);
		break;
	  case SHLD_DAMAGED:
		FNotify(player, "%s shield is damaged and "
		  "may not be raised or lowered.", shields[shield]);
		break;
	}

	return;
}

