/* v0.8
 *
 * commands.c:  Space command interpreter.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "hashtab.h"
#include "commands.h"
#include "smisc.h"
#include "nav.h"
#include "shields.h"
#include "tactical.h"
#include "eng.h"
#include "damage.h"
#include "events.h"
#include "sensors.h"
#include "comm.h"

typedef struct command_entry {

	char 	*name;
	void	(*fun)();	/* handler */
	int 	nargs;
	int 	flags;		/* type of function */

} CMD;

/* type flags */
#define FUNCTION	0x0001		
#define COMMAND		0x0002		
#define C_SHIP		0x0004		/* Caller must be a ship 
					 * (irrelevant w/o C_OBJ below)
					 */
#define C_OBJ		0x0008		/* Caller must be an active object */

#define SCDECL(x)	void x(TAG *, dbref, char **);
#define SFDECL(x)	void x(TAG *, char *, char **, dbref, char **);

dbref parse_target(dbref, char *);
dbref parse_caller(dbref);

SCDECL(cmdAddObject);
SCDECL(cmdRemoveObject);
SCDECL(cmdSetObjectFlag);
SFDECL(funHasObjectFlag);
SCDECL(cmdSetShipFlag);
SFDECL(funHasShipFlag);
SCDECL(cmdGodSpecs);
SCDECL(cmdSpecs);
SCDECL(cmdContactList);
SFDECL(funListContainers);
SCDECL(cmdSetTorpAmmo);
SCDECL(cmdReportContact);
SCDECL(cmdEngToggleSafe);
SCDECL(cmdEngToggleWarn);
SCDECL(cmdEngBatteryOn);
SCDECL(cmdEngBatteryOff);
SCDECL(cmdEngStartReactor);
SCDECL(cmdEngShutdownReactor);
SCDECL(cmdEngAllocate);
SCDECL(cmdEngSetReactor);
SFDECL(funEngReactorSetting);
SFDECL(funEngAllocTac);
SFDECL(funEngAllocNav);
SFDECL(funEngAllocShields);
SFDECL(funEngAllocBatts);
SFDECL(funEngStress);
SFDECL(funEngOutput);
SFDECL(funEngBattStatus);
SFDECL(funEngBattLevel);
SFDECL(funEngBattMax);
SFDECL(funEngBattDisch);
SFDECL(funEngBattMaxDisch);
SCDECL(cmdNavDock);
SCDECL(cmdNavUndock);
SCDECL(cmdNavLand);
SCDECL(cmdNavLaunch);
SCDECL(cmdNavOrbit);
SCDECL(cmdNavSetCourse);
SFDECL(cmdShieldAllocate);
SCDECL(cmdNavSetWarp);
SCDECL(cmdShdAllocate);
SCDECL(cmdShdRaiseShield);
SCDECL(cmdShdLowerShield);
SCDECL(cmdShdCloakOn);
SCDECL(cmdShdCloakOff);
SCDECL(cmdNavOpenDoors);
SCDECL(cmdNavCloseDoors);
SFDECL(funNavHeading);
SFDECL(funNavSpeed);
SFDECL(funShdCloakStatus);
SFDECL(funShdShieldLevel);
SFDECL(funShdShieldMaxLevel);
SFDECL(funShdShieldStatus);
SFDECL(funShdShieldAction);
SFDECL(funNavDoorStatus);
SFDECL(funShdAllocShield);
SCDECL(cmdWarpTable);
SCDECL(cmdTacLock);
SCDECL(cmdTacUnlock);
SCDECL(cmdTacTorpOnline);
SCDECL(cmdTacTorpOffline);
SCDECL(cmdTacGunOnline);
SCDECL(cmdTacGunOffline);
SCDECL(cmdTacFireTorp);
SCDECL(cmdTacFireGun);
SCDECL(cmdTacReloadTorp);
SCDECL(cmdTacAllocate);
SCDECL(cmdTacAutoReload);
SCDECL(cmdTacAutoCharge);

SCDECL(cmdTacTractorLock);
SCDECL(cmdTacTractorUnlock);
SCDECL(cmdTacTractorDock);
SCDECL(cmdTacTractorDraw);
SFDECL(funTacAllocTractor);
SFDECL(funTacTractorTarget);
SFDECL(funTacTractorStatus);

SFDECL(funTacAllocGuns);
SFDECL(funTacAllocTorps);
SFDECL(funTacGunQty);
SFDECL(funTacGunStatus);
SFDECL(funTacGunCharge);
SFDECL(funTacGunPower);
SFDECL(funTacTorpAmmo);
SFDECL(funTacTorpQty);
SFDECL(funTacTorpStatus);
SFDECL(funTacTorpCharge);
SFDECL(funTacTorpPower);
SFDECL(funTacLockStatus);
SFDECL(funTacReloadStatus);
SFDECL(funTacChargeStatus);
SCDECL(cmdCommSetChannel);
SCDECL(cmdCommLabelChannel);
SCDECL(cmdCommSend);
SFDECL(funCommFreq);
SFDECL(funCommLabel);
SCDECL(cmdDamRepair);
SCDECL(cmdDamSystemStatus);
SCDECL(cmdDamTeamStatus);
SCDECL(cmdScan);
SCDECL(cmdSetPosition);
SFDECL(funSensorStatus);
SFDECL(funScannerStatus);
SFDECL(funContactListNumbers);
SFDECL(funContactListDbrefs);
SFDECL(funContactInfoLevel);

SFDECL(funTransListBeamable);
SFDECL(funTransListDests);
SFDECL(funTransPermission);

SFDECL(funRange);
SFDECL(funOwnerNumber);
SFDECL(funFacingShield);

SFDECL(funActive);
SFDECL(funShip);
SFDECL(funWarpCost);
SFDECL(funMaxWarp);
SFDECL(funXYZtoSPH);
SFDECL(funSPHtoXYZ);
SFDECL(funName);
SFDECL(funPosition);
SFDECL(funRelPos);
SFDECL(funLookupContact);
SCDECL(cmdObjList);
SCDECL(cmdSpaceLock);
SCDECL(cmdCauseDamage);
SCDECL(cmdExplosion);
SCDECL(cmdDrainReactor);

/*
 * cmdlist:  List of space functions and commands
 */

CMD cmdlist[] = {
/* Object level commands/functions */
{"add_object", 			cmdAddObject,			1,	COMMAND},
{"rm_object",			cmdRemoveObject,		0,	COMMAND|C_OBJ},
{"set_objflag",			cmdSetObjectFlag,		1,	COMMAND|C_OBJ},
{"set_shipflag",		cmdSetShipFlag,			1,	COMMAND|C_OBJ},
{"objlist",				cmdObjList,				3,	COMMAND},
{"space_lock",			cmdSpaceLock,			2,	COMMAND},

/* Communications commands/functions */

{"comm_set_channel",	cmdCommSetChannel,		2,	COMMAND|C_OBJ|C_SHIP},
{"comm_label_channel",	cmdCommLabelChannel,	2,	COMMAND|C_OBJ|C_SHIP},
{"comm_send",			cmdCommSend,			2,	COMMAND|C_OBJ|C_SHIP},

{"comm_freq",			funCommFreq,			1,	FUNCTION|C_OBJ|C_SHIP},
{"comm_label",			funCommLabel,			1,	FUNCTION|C_OBJ|C_SHIP},

/* Damage control commands/functions */

{"dmg_team_status",		cmdDamTeamStatus,		0,	COMMAND|C_OBJ|C_SHIP},
{"dmg_system_status",	cmdDamSystemStatus,		0,	COMMAND|C_OBJ|C_SHIP},
{"dmg_repair",			cmdDamRepair,			2,	COMMAND|C_OBJ|C_SHIP},

/* Engineering commands/functions */

{"eng_start_reactor",	cmdEngStartReactor,		1,	COMMAND},
{"eng_shutdown_reactor",cmdEngShutdownReactor,	0,	COMMAND|C_OBJ|C_SHIP},
{"eng_set_reactor",		cmdEngSetReactor,		1,	COMMAND|C_OBJ|C_SHIP},
{"eng_battery_on",		cmdEngBatteryOn,		0,	COMMAND|C_OBJ|C_SHIP},
{"eng_battery_off",		cmdEngBatteryOff,		0,	COMMAND|C_OBJ|C_SHIP},
{"eng_toggle_warn",		cmdEngToggleWarn,		0,	COMMAND|C_OBJ|C_SHIP},
{"eng_toggle_safe",		cmdEngToggleSafe,		0,	COMMAND|C_OBJ|C_SHIP},
{"eng_allocate",		cmdEngAllocate,			4,	COMMAND|C_OBJ|C_SHIP},

{"eng_reactor_setting",	funEngReactorSetting,	0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_alloc_tac",		funEngAllocTac,			0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_alloc_nav",		funEngAllocNav,			0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_alloc_batts",		funEngAllocBatts,		0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_alloc_shields",	funEngAllocShields,		0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_stress",			funEngStress,			0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_output",			funEngOutput,			0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_batt_status",		funEngBattStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_batt_level",		funEngBattLevel,		0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_batt_max",		funEngBattMax,			0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_batt_disch",		funEngBattDisch,		0,	FUNCTION|C_OBJ|C_SHIP},
{"eng_batt_max_disch",	funEngBattMaxDisch,		0,	FUNCTION|C_OBJ|C_SHIP},

/* Navigation/Shield commands/functions */

{"nav_dock",			cmdNavDock,				1,	COMMAND|C_OBJ|C_SHIP},
{"nav_undock",			cmdNavUndock,			1,	COMMAND},
{"nav_land",			cmdNavLand,				1,	COMMAND|C_OBJ|C_SHIP},
{"nav_launch",			cmdNavLaunch,			1,	COMMAND},
{"nav_orbit",			cmdNavOrbit,			1,	COMMAND|C_OBJ|C_SHIP},
{"nav_set_course",		cmdNavSetCourse,		3,	COMMAND|C_OBJ|C_SHIP},
{"nav_set_warp",		cmdNavSetWarp,			1,	COMMAND|C_OBJ|C_SHIP},
{"nav_open_doors",		cmdNavOpenDoors,		0,	COMMAND|C_OBJ|C_SHIP},
{"nav_close_doors",		cmdNavCloseDoors,		0,	COMMAND|C_OBJ|C_SHIP},
{"shd_allocate",		cmdShdAllocate,			4,	COMMAND|C_OBJ|C_SHIP},
{"shd_raise_shield",	cmdShdRaiseShield,		1,	COMMAND|C_OBJ|C_SHIP},
{"shd_lower_shield",	cmdShdLowerShield,		1,	COMMAND|C_OBJ|C_SHIP},
{"shd_cloak_on",		cmdShdCloakOn,			0,	COMMAND|C_OBJ|C_SHIP},
{"shd_cloak_off",		cmdShdCloakOff,			0,	COMMAND|C_OBJ|C_SHIP},
	
{"nav_heading",			funNavHeading,			0,	FUNCTION|C_OBJ|C_SHIP},
{"nav_speed",			funNavSpeed,			0,	FUNCTION|C_OBJ|C_SHIP},
{"nav_door_status",		funNavDoorStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"shd_alloc_shield",	funShdAllocShield,		1,	FUNCTION|C_OBJ|C_SHIP},
{"shd_cloak_status",	funShdCloakStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"shd_shield_level",	funShdShieldLevel,		1,	FUNCTION|C_OBJ|C_SHIP},
{"shd_shield_max_level",funShdShieldMaxLevel,	1,	FUNCTION|C_OBJ|C_SHIP},
{"shd_shield_status",	funShdShieldStatus,		1,	FUNCTION|C_OBJ|C_SHIP},
{"shd_shield_action",	funShdShieldAction,		1,	FUNCTION|C_OBJ|C_SHIP},

/* Weapons commands/functions */

{"tac_lock",			cmdTacLock,				1,	COMMAND|C_OBJ},
{"tac_unlock",			cmdTacUnlock,			0,	COMMAND|C_OBJ},
{"tac_torp_online",		cmdTacTorpOnline,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_torp_offline",	cmdTacTorpOffline,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_gun_online",		cmdTacGunOnline,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_gun_offline",		cmdTacGunOffline,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_fire_torp",		cmdTacFireTorp,			5,	COMMAND|C_OBJ|C_SHIP},
{"tac_fire_gun",		cmdTacFireGun,			1,	COMMAND|C_OBJ|C_SHIP},
{"tac_reload_torp",		cmdTacReloadTorp,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_allocate",		cmdTacAllocate,			3,	COMMAND|C_OBJ|C_SHIP},
{"tac_auto_reload",		cmdTacAutoReload,		0,	COMMAND|C_OBJ|C_SHIP},
{"tac_auto_charge",		cmdTacAutoCharge,		0,	COMMAND|C_OBJ|C_SHIP},
{"tac_tractor_lock",	cmdTacTractorLock,		1,	COMMAND|C_OBJ|C_SHIP},
{"tac_tractor_unlock",	cmdTacTractorUnlock,	0,	COMMAND|C_OBJ|C_SHIP},
{"tac_tractor_dock",	cmdTacTractorDock,		0,	COMMAND|C_OBJ|C_SHIP},
{"tac_tractor_draw",	cmdTacTractorDraw,		1,	COMMAND|C_OBJ|C_SHIP},

{"tac_alloc_guns",		funTacAllocGuns,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_alloc_torps",		funTacAllocTorps,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_gun_qty",			funTacGunQty,			0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_gun_status",		funTacGunStatus,		1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_gun_charge",		funTacGunCharge,		1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_gun_power",		funTacGunPower,			1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_torp_qty",		funTacTorpQty,			0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_torp_ammo",		funTacTorpAmmo,			0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_torp_status",		funTacTorpStatus,		1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_torp_charge",		funTacTorpCharge,		1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_torp_power",		funTacTorpPower,		1,	FUNCTION|C_OBJ|C_SHIP},
{"tac_lock_status",		funTacLockStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_reload_status",	funTacReloadStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_charge_status",	funTacChargeStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_alloc_tractor",	funTacAllocTractor,		0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_tractor_target",	funTacTractorTarget,	0,	FUNCTION|C_OBJ|C_SHIP},
{"tac_tractor_status",	funTacTractorStatus,	0,	FUNCTION|C_OBJ|C_SHIP},

/* Transporter functions */

{"trans_list_dests",	funTransListDests,		0,	FUNCTION|C_OBJ},
{"trans_list_beamable",	funTransListBeamable,	0,	FUNCTION|C_OBJ},
{"trans_permission",	funTransPermission,		1,	FUNCTION|C_OBJ},

/* Non console-dependent commands/functions */
{"warp_cost",			funWarpCost,			1,	FUNCTION|C_OBJ|C_SHIP},
{"max_warp",			funMaxWarp,				1,	FUNCTION|C_OBJ|C_SHIP},
{"specs",				cmdSpecs,				0,	COMMAND|C_OBJ},
{"godspecs",			cmdGodSpecs,			0,	COMMAND|C_OBJ},
{"scan",				cmdScan,				1,	COMMAND|C_OBJ|C_SHIP},
{"sensor_status",		funSensorStatus,		0,	FUNCTION|C_OBJ|C_SHIP},
{"scanner_status",		funScannerStatus,	0,	FUNCTION|C_OBJ|C_SHIP},
{"warp_table",			cmdWarpTable,			0,	COMMAND|C_OBJ|C_SHIP},
{"contact_list",		cmdContactList,			2,	COMMAND|C_OBJ}, 
{"report_contact",		cmdReportContact,		1,	COMMAND|C_OBJ},

/* General Coding functions */

{"active",				funActive,				1,	FUNCTION},
{"ship",				funShip,				0,	FUNCTION|C_OBJ},
{"contact_list_num",	funContactListNumbers,	0,	FUNCTION|C_OBJ},
{"contact_list_dbrefs",	funContactListDbrefs,	0,	FUNCTION|C_OBJ},
{"contact_info_level",	funContactInfoLevel,	1,	FUNCTION|C_OBJ},
{"range",				funRange,				1,	FUNCTION|C_OBJ},
{"owner_number",		funOwnerNumber,			0,	FUNCTION|C_OBJ|C_SHIP},
{"facing_shield",		funFacingShield,		1,	FUNCTION|C_OBJ},
{"pos",					funPosition,			0,	FUNCTION|C_OBJ},
{"set_pos",				cmdSetPosition,			3,	COMMAND|C_OBJ},
{"rel_pos",				funRelPos,				1,	FUNCTION|C_OBJ},
{"lookup_contact",		funLookupContact,		1,	FUNCTION|C_OBJ},
{"has_obj_flag",		funHasObjectFlag,		1,	FUNCTION|C_OBJ},
{"has_ship_flag",		funHasShipFlag,			1,	FUNCTION|C_OBJ|C_SHIP},
{"cause_damage",		cmdCauseDamage,			3,	COMMAND|C_OBJ},
{"explosion",			cmdExplosion,			5,	COMMAND},
{"drain_reactor",		cmdDrainReactor,		1,	COMMAND|C_OBJ|C_SHIP},
{"name",				funName,				0,	FUNCTION|C_OBJ},
{"list_containers",		funListContainers,		0,	FUNCTION|C_OBJ},
{"set_torp_ammo",		cmdSetTorpAmmo,			1,	COMMAND|C_OBJ|C_SHIP},

/* Extra functions */
{"xyz_to_sph",			funXYZtoSPH,			3,	FUNCTION},
{"sph_to_xyz",			funSPHtoXYZ,			3,	FUNCTION},

/* End-of-list marker */
{"",					NULL,					0}};

HNODE *fun_htab;

void cmdInitFunctionTable(void)
{
	int i;

	fun_htab = NULL;
	for (i=0; *(cmdlist[i].name) != '\0'; i++) 
		hashInsert(cmdlist[i].name, (int *)(cmdlist + i), &fun_htab);

	return;
}	

MUDFUNCTION(fun_spacecall)
{
	char wbuf[SMALL_BUF_SIZE];
	dbref data_object;
	int offset = 0;
	TAG *object = NULL;
	CMD *fp;

	if (NumArgs < 1) {
		WriteFunctionBuffer("#-1 FUNCTION(SC) REQUIRES AT LEAST ONE ARGUMENT");
		return;
	}

	fp = (CMD *)hashSearch(Args[0], fun_htab);
	if (fp == NULL) { 
		WriteFunctionBuffer("#-1 INVALID SPACE FUNCTION"); 
		return; 
	} 

	if (NumArgs < fp->nargs + 1) { 
		sprintf(wbuf, "#-1 SPACE FUNCTION(%s) REQUIRES %d ARGUMENTS", 
		  fp->name, fp->nargs); 
		WriteFunctionBuffer(wbuf); 

		return;
	}

	if ((fp->flags & C_OBJ) != 0) {

		if (NumArgs == fp->nargs + 2) {
			data_object = parse_target(Player, Args[1]);
			offset = 1;
		}
		else
			data_object = parse_caller(Player);

		if (!ValidObject(data_object)) {
			WriteFunctionBuffer("#-1 BAD DATA OBJECT");
			return;
		}

		object = objFindObject(data_object);

		if ((object==NULL) || Removed(object)) {
			if ((fp->flags & C_SHIP) != 0)
				Notify(Cause, "The ship must be active to use"
				  " that command.");
			else
				Notify(Cause, "Object not in space.");

			return;
		}

		if (object != NULL)
			if ((fp->flags & C_SHIP) != 0 && !Ship(object)) {
				Notify(Cause, "The object must be a ship to"
				  " use that command.");
				return;
			}
	}

	/* get the dref of the calling object */

	if ((fp->flags & COMMAND) != 0)
		CALL_COMMAND(fp->fun, object, Cause, Args + offset);

	else if ((fp->flags & FUNCTION) != 0)
		CALL_FUNCTION(fp->fun, object, Cause, Args + offset);

	return;

}

/*
 * parse_caller:  If DBREFDATA is set on the calling object, assume that 
 *		it is a console of some sort, and use the dbref in the field
 *		as the data object.  Otherwise, use the calling object's
 *		dbref.
 */
dbref parse_caller(dbref caller)
{
	dbref data;

	data = dbrefData(caller);

	if (data == NOTHING)
		data = caller;

	return data;
}
	
/*
 * parse_target:  A target has been provided.  If the target starts with
 *		a "C", it is a contact, and we should look for the contact
 *		number relative to the caller (see parse_caller above).
 *		If the target isn't a contact, assume it's another data object.
 *		
 */
dbref parse_target(dbref data, char *arg)
{
	TAG *object;
	CONTACT *contact;

	if (*arg == 'c' || *arg == 'C')	{
		if ((object = objFindObject(parse_caller(data)))==NULL)
			return NOTHING;
		
		if ((contact = snsFindContactByNumber(object, atoi(arg+1)))==NULL)
			return NOTHING;

		return contact->listref->data_object;
	}
	else
		return (parseDbref(arg));
}

/*
 * Space god commands
 */

void cmdAddObject(TAG *object, dbref player, char **args)
{
	int data;

	data = parseDbref(args[1]);

	if (!ValidObject(data)) {
		Notify(player, "That is not a valid space object.");
		return;
	}

	objAddObject(parseDbref(args[1]));
}

void cmdRemoveObject(TAG *object, dbref player, char **args)
{
	objRemoveObject(object);
}
	
void cmdSetObjectFlag(TAG *object, dbref player, char **args)
{

	if (strlen(args[1]) > 0) {
		if (*(args[1]) == '!')
			objSetObjectFlag(object, player, 0, args[1]+1);
		else
			objSetObjectFlag(object, player, 1, args[1]);

	}
	else
		Notify(player, "Invalid flag.");

	return;

}

void funHasObjectFlag(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", objHasObjectFlag(object, player, args[1]));
}

void cmdSetShipFlag(TAG *object, dbref player, char **args)
{

	if (strlen(args[1]) > 0) {
		if (*(args[1]) == '!')
			objSetShipFlag(object, player, 0, args[1]+1);
		else
			objSetShipFlag(object, player, 1, args[1]);

	}
	else
		Notify(player, "Invalid flag.");

	return;

}

void funHasShipFlag(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", objHasShipFlag(object, player, args[1]));
}

void cmdGodSpecs(TAG *object, dbref player, char **args)
{
	objDisplaySpecs(object, player, 1);
}

void cmdObjList(TAG *object, dbref player, char **args)
{
	int space;

	space = atoi(args[1]);

	if (space < 0 || space >= NUM_SPACES) {
		Notify(player, "Invalid space.");
		return;
	}

	objList(player, space, atoi(args[2]), atoi(args[3]));
}

void cmdSpaceLock(TAG *object, dbref player, char **args)
{
	int space;
	
	space = atoi(args[1]);

	if (space < 0 || space >= NUM_SPACES) {
		FNotify(player, "Valid spaces are between 0 and %d.",
			NUM_SPACES - 1);
		return;
	}

	space_lock[space]=atoi(args[2]);

	if (space_lock[space])
		Notify(player, "Locked.");
	else
		Notify(player, "Unlocked.");

	return;

}

void cmdContactList(TAG *object, dbref player, char **args)
{
	snsDisplayContacts(object, player, atoi(args[1]), atoi(args[2]));	
}

void cmdReportContact(TAG *object, dbref player, char **args)
{
	CONTACT *contact;

	contact = snsFindContactByNumber(object, atoi(args[1]));

	if (contact == NULL)
		Notify(player, "Contact not on sensors.");
	else
		snsDisplaySensorInfo(object, contact, player, DSD_ROUTINE);

}

/* 
 * Basic informative commands
 */

void cmdSpecs(TAG *object, dbref player, char **args)
{
	objDisplaySpecs(object, player, 0);
}

/*
 * Engineering commands
 */

void cmdEngToggleSafe(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;

	ship->eng_safeties = !ship->eng_safeties;

	if (ship->eng_safeties)
		Notify(player, "Safety overrides enabled.");
	else
		Notify(player, "Safety overrides disabled.");

	return;
}

void cmdEngToggleWarn(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;

	ship->eng_warnings = !ship->eng_warnings;

	if (ship->eng_warnings)
		Notify(player, "Warning messages enabled.");
	else
		Notify(player, "Warning messages disabled.");

	return;
}

void cmdEngBatteryOn(TAG *object, dbref player, char **args)
{
	engBatteryOn(object, player);
}

void cmdEngBatteryOff(TAG *object, dbref player, char **args)
{
	engBatteryOff(object, player);
}

void cmdEngStartReactor(TAG *object, dbref player, char **args)
{
	dbref data;

	data = parseDbref(args[1]);

	if (objFindObject(data) == NULL)
		engStartReactor(object, player, parseDbref(args[1]));
	else
		Notify(player, "The engine is already running.");
}

void cmdEngShutdownReactor(TAG *object, dbref player, char **args)
{
	engShutdownReactor(object, player);
}

void cmdEngAllocate(TAG *object, dbref player, char **args)
{
	engAllocate(object, player, atoi(args[1]), atoi(args[2]), atoi(args[3]), atoi(args[4]));
}

void cmdEngSetReactor(TAG *object, dbref player, char **args)
{
	engSetReactorOutput(object, player, atoi(args[1]));
}

void funEngReactorSetting(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->reactor_setting);
}

void funEngAllocTac(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->alloc_tac);
}

void funEngAllocNav(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->alloc_nav);
}

void funEngAllocBatts(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->alloc_batt);
}

void funEngAllocShields(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->alloc_shields);
}

void funEngStress(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", 100 * object->shipdata->overload_points /
	  object->shipdata->max_overload_points);
}

void funEngOutput(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->reactor_output);
}

void funEngBattStatus(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{

	switch (object->shipdata->battery_status) {

	  case BTTY_OFFLINE:
		WriteFunctionBuffer("OFF");
		break;
	  case BTTY_ONLINE:
		WriteFunctionBuffer("ON");
		break;
	  case BTTY_DAMAGED:
		WriteFunctionBuffer("DAMAGED");
		break;
	}
}

void funEngBattLevel(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->battery_level);
}

void funEngBattMax(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->battery_capacity);
}

void funEngBattDisch(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->battery_discharge);
}

void funEngBattMaxDisch(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->battery_discharge_max);
}

/*
 * Navigation commands
 */

void cmdNavDock(TAG *object, dbref player, char **args)
{
	CONTACT *dock;

	dock = snsFindContactByNumber(object, atoi(args[1]));

	if (dock==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	navDock(object, player, dock->listref, 0);
}

void cmdNavUndock(TAG *object, dbref player, char **args)
{
	dbref data;

	data = parseDbref(args[1]);

	if (objFindObject(data) == NULL)
		navUndock(object, player, parseDbref(args[1]));	
	else
		Notify(player, "The ship isn't currently docked.");
}

void cmdNavLand(TAG *object, dbref player, char **args)
{
	navLand(object, player, atoi(args[1]));
}

void cmdNavLaunch(TAG *object, dbref player, char **args)
{
	dbref data;

	data = parseDbref(args[1]);

	if (objFindObject(data) == NULL)
		navLaunch(object, player, parseDbref(args[1]));	
	else
		Notify(player, "The ship is not on a planet.");
}

void cmdNavOrbit(TAG *object, dbref player, char **args)
{
	navOrbit(object, player, atoi(args[1]));
}

void cmdNavSetCourse(TAG *object, dbref player, char **args)
{
	float elevation;

	elevation = atof(args[3]);

	if (strlen(args[2]) && *(args[2])=='-')
		elevation = -elevation;

	navSetCourse(object, player, atof(args[1]), elevation);
}

void cmdShdAllocate(TAG *object, dbref player, char **args)
{
	shdAllocate(object, player, atoi(args[1]), atoi(args[2]), atoi(args[3]),
	  atoi(args[4]));
}

void cmdNavSetWarp(TAG *object, dbref player, char **args)
{
	navSetWarp(object, player, atof(args[1]));
}

void cmdShdRaiseShield(TAG *object, dbref player, char **args)
{
	shdRaiseShield(object, player, atoi(args[1]));
}

void cmdShdLowerShield(TAG *object, dbref player, char **args)
{
	shdLowerShield(object, player, atoi(args[1]));
}

void cmdShdCloakOn(TAG *object, dbref player, char **args)
{
	shdCloakOn(object, player);
}

void cmdShdCloakOff(TAG *object, dbref player, char **args)
{
	shdCloakOff(object, player);
}

void cmdNavOpenDoors(TAG *object, dbref player, char **args)
{
	navDoorControl(object, player, 1);
}

void cmdNavCloseDoors(TAG *object, dbref player, char **args)
{
	navDoorControl(object, player, 0);
}

void funPosition(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	char wbuf[SMALL_BUF_SIZE];

	sprintf(wbuf, "%d %d %d", object->pos.x, object->pos.y, object->pos.z);

	WriteFunctionBuffer(wbuf);
}

void cmdSetPosition(TAG *object, dbref player, char **args)
{
	object->pos.x = atoi(args[1]);
	object->pos.y = atoi(args[2]);
	object->pos.z = atoi(args[3]);
}

void funRelPos(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	char wbuf[SMALL_BUF_SIZE];
	SPH bearing;
	XYZ rel_pos;
	TAG *target;

	if ((target=objFindObject(parseDbref(args[1])))==NULL) {
		Notify(player, "Invalid target!");
		return;
	}	

	rel_pos.x = target->pos.x - object->pos.x;
	rel_pos.y = target->pos.y - object->pos.y;
	rel_pos.z = target->pos.z - object->pos.z;

	xyz_to_sph(rel_pos, &bearing);
	sprintf(wbuf, "%1.2f %+2.2f %d", bearing.bearing, bearing.elevation,
	  bearing.range);

	WriteFunctionBuffer(wbuf);
}

void funLookupContact(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	CONTACT *contact;

	contact = snsFindContactByNumber(object, atoi(args[1]));

	if (contact != NULL)
		FWriteFunctionBuffer("#%d", contact->listref->data_object);
	else
		WriteFunctionBuffer("#-1");
}	

void funNavHeading(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	char wbuf[SMALL_BUF_SIZE];

	sprintf(wbuf, "%1.2f %+2.2f", object->heading.bearing, 
	  object->heading.elevation);

	WriteFunctionBuffer(wbuf);
}
	
void funNavSpeed(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%3.2f",object->speed);
}

void funShdCloakStatus(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	switch (object->shipdata->cloak_status) {

	  case CLOAK_NONE:
		WriteFunctionBuffer("NONE");
		break;
	  case CLOAK_OFF:
		WriteFunctionBuffer("OFF");
		break;
	  case CLOAK_ON:
		WriteFunctionBuffer("ON");
		break;
	  case CLOAKING:
		WriteFunctionBuffer("CLOAKING");
		break;
	  case DECLOAKING:
		WriteFunctionBuffer("DECLOAKING");
		break;
	}
}
			
void funShdShieldLevel(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	int shield;

	shield = atoi(args[1]);

	if (shield >= 0 && shield <= 3) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->shield_level[shield]);

}

void funShdShieldMaxLevel(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	int shield;

	shield = atoi(args[1]);

	if (shield >= 0 && shield <= 3) 
		FWriteFunctionBuffer("%d",
		  object->shipdata->shield_max[shield]);

}

void funShdShieldStatus(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	int shield;

	shield = atoi(args[1]);

	if (shield >= 0 && shield <= 3)
		switch (object->shipdata->shield_status[shield]) {

		  case SHLD_READY:
			WriteFunctionBuffer("READY");
			break;
		  case SHLD_UP:
			WriteFunctionBuffer("UP");
			break;
		  case SHLD_DAMAGED:
			WriteFunctionBuffer("DAMAGED");
			break;
		}

}
		
void funShdShieldAction(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	int shield;

	shield = atoi(args[1]);

	if (shield >= 0 && shield <= 3) {

		if (object->shipdata->shield_status[shield] !=  SHLD_DAMAGED)
			switch (object->shipdata->shield_action[shield]) {

			  case SHLD_STABLE:
				WriteFunctionBuffer("STABLE");
				break;
			  case SHLD_CHARGING:
				WriteFunctionBuffer("CHARGE");
				break;
			  case SHLD_FALLING:
				WriteFunctionBuffer("FALL");
				break;
			  case SHLD_OVERLOADED:
				WriteFunctionBuffer("OVERLOAD");
				break;
			}

	}

}

void funNavDoorStatus(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	switch (object->shipdata->door_status) {
		
	  case DOORS_NONE:
		WriteFunctionBuffer("NONE");
		break;
	  case DOORS_CLOSED:
		WriteFunctionBuffer("CLOSED");
		break;	
	  case DOORS_OPEN:
		WriteFunctionBuffer("OPEN");
		break;
	}

}

void funShdAllocShield(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	int shield;

	shield = atoi(args[1]);

	if (shield >= 0 && shield <= 3) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->shield_alloc[shield]);
}

void funActive(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", (objFindObject(parseDbref(args[1])) == NULL)
	  ? 0 : 1);
}

void funShip(TAG *object, char *buff, char **bufc, dbref player,
  char **args)
{
	FWriteFunctionBuffer("%d", Ship(object) ? 1 : 0);
}

void cmdWarpTable(TAG *object, dbref player, char **args)
{
	float setting;

	Notify(player, "Warp costs table:");
	Notify(player, " ");

	for (setting = 1.0; setting <= 10.0; setting++) 
		FNotify(player, "Warp %3.1f costs %d.", setting, 
		  navWarpCost(object, setting));

}

void funWarpCost(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{

	FWriteFunctionBuffer("%d", navWarpCost(object, atof(args[1])));
}

void funMaxWarp(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%3.1f", navMaxWarp(object, atoi(args[1])));
}

void funSPHtoXYZ(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	char wbuf[SMALL_BUF_SIZE];
	SPH sph;
	XYZ xyz;

	sph.bearing = atof(args[1]);
	sph.elevation = atof(args[2]);
	sph.range = atoi(args[3]);

	sph_to_xyz(sph, &xyz);

	sprintf(wbuf, "%d %d %d", xyz.x, xyz.y, xyz.z);
	WriteFunctionBuffer(wbuf);
}

void funXYZtoSPH(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	char wbuf[SMALL_BUF_SIZE];
	SPH sph;
	XYZ xyz;

	xyz.x = atoi(args[1]);
	xyz.y = atoi(args[2]);
	xyz.z = atoi(args[3]);

	xyz_to_sph(xyz, &sph);

	sprintf(wbuf, "%1.2f %+1.2f %d", sph.bearing, sph.elevation, sph.range);
	WriteFunctionBuffer(wbuf);
}

void cmdTacAutoCharge(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;
	int i;

	ship->auto_online = !ship->auto_online;

	if (ship->auto_online) {

		FNotify(player, "%s charging controls set to automatic.",
		  capstr(ship->torp_string));
		for (i=0; i < ship->number_of_torps; i++) {
			if (ship->torp[i].status == TORP_LOADED) {
				FNotify(player, "Bringing %s tube %d online:",
				  ship->torp_string, i);
			  	tacTorpControl(object, player, i, 1);
			}
		}

	}
	else
		FNotify(player, "%s charging controls set to manual.",
		  capstr(ship->torp_string));

	return;
} 

void cmdTacAutoReload(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;
	int i;

	ship->auto_reload = !ship->auto_reload;

	if (ship->auto_reload) {
		FNotify(player, "%s auto reloading enabled.", 
		  capstr(ship->torp_string));

		for (i=0; i < ship->number_of_torps && 
		  ship->free_torp_loaders; i++) {
			if (ship->torp[i].status == TORP_EMPTY) {
				tacReloadTorp(object, player, i);
				break;
			}
		}
	}
	else
		FNotify(player, "%s auto reloading disabled.",
		  capstr(ship->torp_string));

	return;
}

void cmdTacFireGun(TAG *object, dbref player, char **args)
{
	int nfire, tdam, withc;
	int bank, damage = 0, fireall, i;
	TAG *target = NULL;
	SHIP *ship = object->shipdata;
	CONTACT *us;

	fireall = strcmp(args[1], "all") ? 0 : 1;
	bank = atoi(args[1]);

	if (object->locked_on == NULL) {
		FNotify(player, "Weapons must be locked in order to fire %ss.",
		  object->shipdata->gun_string);
		return;
	}
		
	target = object->locked_on->listref;

	if (Pacifist(ship)) {
		Notify(player, "OOC:  Weapon use is presently disabled.");
		return;
	}

	if (!fireall && (bank < 0 || bank >= ship->number_of_guns)) {
		FNotify(player, "Invalid %s number.", ship->gun_string);
		return;
	}

	/* nfire is the number of guns that fired. withc the number that
	   were charged. */
	nfire = withc = 0;

	if (fireall) {
		for (i=0; i < ship->number_of_guns; i++) 
			if (ship->gun[i].charge > 0) {
				withc++;
				tdam = tacFireGun(object, player, i);
				if (tdam >= 0) {
					damage += tdam;
					nfire ++;
				}
			}
	} else {
		if (ship->gun[bank].charge > 0) {
			withc++;
			tdam = tacFireGun(object, player, bank);
			if (tdam >= 0) {
				damage = tdam;
				nfire++;
			}
		}
	}

	/* If no weapons were charged, we need to shout about it */
	if (withc == 0) {
		FNotify(player, "%s must be charged before firing.", 
		  capstr(ship->gun_string));
		return;
	}

	/* We didn't actually fire any weapons - maybe cause we're disabled,
	   or out of range, or something. The message already went out to
	   the user about this problem in tacFireGun. */
	if (nfire == 0)
		return;

	/* At this point, we know at least 1 gun actually fired, and so we
	   must do all the emits about who fired what at whom */
	us = snsFindContact(target, object);

	if (Ship(target)) {
		if (us != NULL) 
			FNotify(dbrefUser(target->shipdata->tactical), 
			  "Incoming %s fire from contact [%d]: %s.",
			  ship->gun_string, us->contact_number,
			  object->name);
		else
			FNotify(dbrefUser(target->shipdata->tactical),
			  "Incoming %s fire from an unknown source!",
			  ship->gun_string);

	}

	evAttacked(target, us);
	evBroadcast(object, target, "fired upon", EVENT_ATTACKED_OTHER);

	if (object->space == 0)
		log_space("%s (#%d) fired guns at %s (#%d) for %d damage from "
		  "range %d.", object->name, object->data_object, target->name,
		  target->data_object, damage, distance(object->pos, 
		  target->pos));

	if (damage > 0)
		damBattleDamage(object->pos, target, damage, WPN_GUN);

	return;

}

void cmdTacFireTorp(TAG *object, dbref player, char **args)
{
	TAG *target = NULL;
	int tube;
	int hits = 0, fired = 0, damage = 0, i;
	int canshoot = 0;
	int fireall, nolock;
	SHIP *ship = object->shipdata;
	CONTACT *us;
	SPH direction;
	XYZ rel_pos;

	fireall = strcmp(args[1], "all") ? 0 : 1;
	tube = atoi(args[1]);

	if (strcmp(args[2], "NONE") != 0) {
		nolock = 1;
		direction.bearing = atof(args[2]);
		direction.elevation = atof(args[4]);

		if (strlen(args[4]) && *(args[3])=='-')
			direction.elevation = -direction.elevation;
		direction.range = atoi(args[5]);
	}
	else {
		nolock = 0;
		direction.range = 0;
	}

	if (Cloaked(object)) {
		Notify(player, "You cannot fire while cloaked.");
		return;
	}
	
	if (Pacifist(object->shipdata)) {
		Notify(player, "OOC:  Weapon use is presently disabled.");
		return;
	}

	if (!fireall && (tube < 0 || tube >= ship->number_of_torps)) {
		FNotify(player, "Invalid %s number.", ship->torp_string);
		return;
	}

	if (fireall) {
		for (i=0; i < ship->number_of_torps; i++)
			if (ship->torp[i].status == TORP_ARMED)
				canshoot = 1;

		if (!canshoot) {
			FNotify(player, "At least one %s must be armed before "
			  "firing.", ship->torp_string);
			return;
		}
	}
	else {
		if (ship->torp[tube].status != TORP_ARMED) {
			FNotify(player, "%s must be armed before firing.", 
			  capstr(ship->torp_string));
			return;
		}
	}

	if (object->locked_on == NULL) {
		if (!nolock) {
			FNotify(player, "A heading must be specified if %ss "
			  "are to be fired without lock.", 
			  object->shipdata->torp_string);
			return;
		}
			
	}

	if (!nolock) {
		target = object->locked_on->listref;
		rel_pos.x = target->pos.x - object->pos.x;	
		rel_pos.y = target->pos.y - object->pos.y;	
		rel_pos.z = target->pos.z - object->pos.z;
		xyz_to_sph(rel_pos, &direction);
		direction.range = 0;
	}

	tacInitFireTable(object, direction);

	if (fireall) {
		for (i=0; i < ship->number_of_torps; i++) 
			if (ship->torp[i].status == TORP_ARMED) {
				fired++;
				if (tacFireTorp(object, player, i, direction,
				  nolock) > 0) {
					hits++;
					damage += ship->torp[i].power;
				}
			}
	}
	else {
		fired = 1;
		if (tacFireTorp(object, player, tube, direction, nolock)) {
			damage += ship->torp[tube].power;
			hits++;
		}
	}

	tacFreeFireTable();

	/* No messages or events if no torps were fired. This should never
	   happen due to the checking earlier on. */
	if (fired == 0)
		return;

	if (!nolock) {
		us = snsFindContact(target, object);

		if (Ship(target)) {
			if (us != NULL) 
				FNotify(dbrefUser(target->shipdata->tactical), 
				  "Incoming %s%s from contact [%d]: %s.",
				  ship->torp_string, fireall ? "s fired" : 
				  " fired", us->contact_number, object->name);
			else
				FNotify(dbrefUser(target->shipdata->tactical),
				  "Incoming %s%s from an unknown source!",
				  ship->gun_string, fireall ? "s fired" : 
				  " fired");
		}

		evAttacked(target, us);
		evBroadcast(object, target, "fired upon", 
		  EVENT_ATTACKED_OTHER);

	}
	else
		evBroadcast(object, NULL, "fired upon an unknown target",
		  EVENT_ATTACKED_OTHER);

	if (!nolock) {
		if (Ship(target))
			FNotify(dbrefUser(target->shipdata->tactical),
			  "%d/%d enemy %ss hit.", hits, fireall ? fired : 1,
			  ship->torp_string);

		if (object->space == 0)
			log_space("%s (#%d) hit %s (#%d) with %d/%d torps for "
			  "%d damage from range %d.", object->name, 
			  object->data_object, target->name, 
			  target->data_object, hits, fired, damage, 
			  distance(object->pos, target->pos));

		if (hits > 0)
			damBattleDamage(object->pos, target, damage, WPN_TORP);
	}

	return;
}

void cmdTacLock(TAG *object, dbref player, char **args)
{
	tacLockWeapons(object, player, atoi(args[1]));	
}

void cmdTacUnlock(TAG *object, dbref player, char **args)
{
	tacUnlock(object, player);
}

void cmdTacTractorLock(TAG *object, dbref player, char **args)
{
	tacTractorLock(object, player, atoi(args[1]));	
}

void cmdTacTractorUnlock(TAG *object, dbref player, char **args)
{
	tacTractorUnlock(object, player);
}

void cmdTacTractorDock(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;

	switch (ship->tractor_status) {
	  case TRACTOR_NONE:
		Notify(player, "This ship is not equipped with a tractor beam "
		  "generator.");
		break;

	  case TRACTOR_OFF:
	  case TRACTOR_DAMAGED:
		Notify(player, "The tractor beam is not presently engaged.");
		break;

	  default:
		Notify(player, "Attempting to pull the tractor beam target into"
		  " dock.");
		navDock(object->shipdata->tractor_target, player, object, 1);
		break;
	}
}

void cmdTacTractorDraw(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;
	int action;

	action = atoi(args[1]);

	switch (ship->tractor_status) {
	  case TRACTOR_NONE:
		Notify(player, "This ship is not equipped with a tractor beam "
		  "generator.");
		break;

	  case TRACTOR_OFF:
	  case TRACTOR_DAMAGED:
		Notify(player, "The tractor beam is not presently engaged.");
		break;

	  case TRACTOR_ON:	
		if (action) {
			Notify(player, "Tractor beam draw mode engaged.");
			ship->tractor_status = TRACTOR_DRAW;
		}
		else
			Notify(player, "Tractor beam draw mode is not "
			  "engaged.");
		break;

	  case TRACTOR_DRAW:
		if (!action) {
			Notify(player, "Tractor beam draw mode disengaged.");
			ship->tractor_status = TRACTOR_ON;
		}
		else
			Notify(player, "Tractor beam draw mode is already "
			  "engaged.");
		break;
	}

	return;
}

void cmdTacTorpOnline(TAG *object, dbref player, char **args)
{
	tacTorpControl(object, player, atoi(args[1]), 1);
}

void cmdTacTorpOffline(TAG *object, dbref player, char **args)
{
	tacTorpControl(object, player, atoi(args[1]), 0);
}

void cmdTacGunOnline(TAG *object, dbref player, char **args)
{
	tacGunControl(object, player, atoi(args[1]), 1);
}
	
void cmdTacGunOffline(TAG *object, dbref player, char **args)
{
	tacGunControl(object, player, atoi(args[1]), 0);
}

void cmdTacReloadTorp(TAG *object, dbref player, char **args)
{
	tacReloadTorp(object, player, atoi(args[1]));
}

void cmdTacAllocate(TAG *object, dbref player, char **args)
{
	tacAllocate(object, player, atoi(args[1]), atoi(args[2]), 
	  atoi(args[3]));
}

void cmdScan(TAG *object, dbref player, char **args)
{
	snsScan(object, player, atoi(args[1]));
}

void funTacAllocGuns(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->talloc_guns);
}

void funTacAllocTorps(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->talloc_torps);
}

void funTacAllocTractor(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->talloc_tractor);
}

void funTacGunQty(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->number_of_guns);
}

void funTacGunStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int gun;

	gun = atoi(args[1]);
	if (gun >= 0 && gun < object->shipdata->number_of_guns) 
		switch (object->shipdata->gun[gun].status) {
		  case GUN_ONLINE:
			WriteFunctionBuffer("ON");
			break;	
		  case GUN_OFFLINE:
			WriteFunctionBuffer("OFF");
			break;	
		  case GUN_DAMAGED:
			WriteFunctionBuffer("DAMAGED");
			break;	
		}
}

void funTacGunCharge(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int gun;

	gun = atoi(args[1]);
	if (gun >= 0 && gun < object->shipdata->number_of_guns) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->gun[gun].charge);

}

void funTacGunPower(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int gun;

	gun = atoi(args[1]);
	if (gun >= 0 && gun < object->shipdata->number_of_guns) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->gun[gun].power);

}

void funTacTorpQty(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->number_of_torps);
}

void funTacTorpAmmo(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->torp_ammo);
}

void funTacTorpStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int tube;

	tube = atoi(args[1]);
	if (tube >= 0 && tube < object->shipdata->number_of_torps) 
		switch (object->shipdata->torp[tube].status) {
		  case TORP_EMPTY:
			WriteFunctionBuffer("EMPTY");
			break;	
		  case TORP_LOADING:
			WriteFunctionBuffer("RELOADING");
			break;	
		  case TORP_LOADED:
			WriteFunctionBuffer("LOADED");
			break;	
		  case TORP_ONLINE:
			WriteFunctionBuffer("CHARGING");
			break;	
		  case TORP_ARMED:
			WriteFunctionBuffer("ARMED");
			break;	
		  case TORP_DAMAGED:
			WriteFunctionBuffer("DAMAGED");
			break;	
		}
}

void funTacTorpCharge(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int tube;

	tube = atoi(args[1]);
	if (tube >= 0 && tube < object->shipdata->number_of_torps) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->torp[tube].charge);

}

void funTacTorpPower(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int tube;

	tube = atoi(args[1]);
	if (tube >= 0 && tube < object->shipdata->number_of_torps) 
		FWriteFunctionBuffer("%d", 
		  object->shipdata->torp[tube].power);

}

void funTacLockStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	if (object->locked_on) 
		FWriteFunctionBuffer("LOCKED %d", 
		  object->locked_on->contact_number);
	else if (object->pending_lock)
		FWriteFunctionBuffer("PENDING %d",
		  object->pending_lock->contact_number);
	else
		WriteFunctionBuffer("OFF");
}

void funSensorStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	if (object->shipdata->damage[SYSTEM_SENSORS].status > '6')
		WriteFunctionBuffer("DAMAGED");
	else
		WriteFunctionBuffer("OK");
}
		
void funScannerStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	if (object->shipdata->damage[SYSTEM_SCANNERS].status > '6')
		WriteFunctionBuffer("DAMAGED");
	else
		WriteFunctionBuffer("OK");
}

void funTacReloadStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	if (object->shipdata->auto_reload)
		WriteFunctionBuffer("ON");
	else
		WriteFunctionBuffer("OFF");
}

void funTacChargeStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	if (object->shipdata->auto_online)
		WriteFunctionBuffer("ON");
	else
		WriteFunctionBuffer("OFF");
}

void funTacTractorTarget(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	CONTACT *contact;

	contact = snsFindContact(object, object->shipdata->tractor_target);

	if (contact != NULL)
		FWriteFunctionBuffer("%d", contact->contact_number);

}

void funTacTractorStatus(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	SHIP *ship = object->shipdata;

	switch (ship->tractor_status) {

	  case TRACTOR_NONE:
		WriteFunctionBuffer("NONE");
		break;	
	  case TRACTOR_OFF:
		WriteFunctionBuffer("OFF");
		break;	
	  case TRACTOR_ON:
		WriteFunctionBuffer("ON");
		break;	
	  case TRACTOR_DRAW:
		WriteFunctionBuffer("DRAW");
		break;
	  case TRACTOR_DAMAGED:
		WriteFunctionBuffer("DAMAGED");
		break;
	}
	return;
}

void cmdCauseDamage(TAG *object, dbref player, char **args)
{

	TAG *target;
	int damage;
	int type;
	char *dstr;

	target = objFindObject(parseDbref(args[1]));
	damage = atoi(args[2]);
	type = atoi(args[3]);

	if (target==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (object->space == 0) {
		switch (type) {
		  case 0: 
			dstr="gun";
			break;
		  case 1: 
			dstr="torp";
			break;
		  case 2: 
			dstr="explosion";
			break;
		  case 3: 
			dstr="";
			break;
		  default:
			dstr="<unknown>";
			break;
		}

		log_space("%s (#%d) hits %s (#%d) with %d points of %s damage "
		  "from range %d.", object->name, object->data_object, 
		  target->name, target->data_object, damage, dstr, 
		  distance(object->pos, target->pos));

	}

	damBattleDamage(object->pos, target, atoi(args[2]), atoi(args[3]));

}

void cmdExplosion(TAG *object, dbref player, char **args)
{

	int damage, space;
	XYZ point;

	space = atoi(args[1]);

	if (space < 0 || space >= NUM_SPACES) {
		FNotify(player, "Valid spaces are between 0 and %d.",
			NUM_SPACES - 1);
		return;
	}
		
	point.x = atoi(args[2]);
	point.y = atoi(args[3]);
	point.z = atoi(args[4]);
	damage = atoi(args[5]);

	damExplosionDamage(space, point, damage);
	return;
}

void cmdDrainReactor(TAG *object, dbref player, char **args)
{
	SHIP *ship = object->shipdata;
	int power;
	int old_drain;

	power = atoi(args[1]);

	old_drain = object->shipdata->reactor_drain;

	if (power >= 0) {
		object->shipdata->reactor_drain = power;
		FNotify(dbrefUser(ship->eng),
		  "Engine power %s.  New maximum output is %d.",
		  power > old_drain ? "draining" : "restored",
		  ship->current_reactor_output_max - ship->reactor_drain);
		engAllocCheck(object);
		evReactorDrain(object);
	}

	return;
}	

void funName(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%s", object->name);
}

void funListContainers(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	evListContainers(object, buff);
	*bufc = buff + strlen(buff);
}

void cmdSetTorpAmmo(TAG *object, dbref player, char **args)
{
	object->shipdata->torp_ammo = atoi(args[1]);
}

void cmdCommSetChannel(TAG *object, dbref player, char **args)
{
	int channel, freq;

	channel = atoi(args[1]);
	freq = atoi(args[2]);

	if (channel < 0 || channel >= MAX_CHANNELS) {
		Notify(player, "Valid channels are between 0 and 5 inclusive.");
		return;
	}

	if (freq < 0 || freq > 65536) {
		Notify(player, "Valid frequencies are between 1 and 65536.");
		return;
	}

	commSetChannel(object, player, channel, freq);

	return;
}

void cmdCommLabelChannel(TAG *object, dbref player, char **args)
{
	int channel;

	channel = atoi(args[1]);

	if (channel < 0 || channel >= MAX_CHANNELS) {
		Notify(player, "Valid channels are between 0 and 5 inclusive.");
		return;
	}

	commLabelChannel(object, player, channel, args[2]);

	return;
}

void cmdCommSend(TAG *object, dbref player, char **args)
{
	int channel;

	channel = atoi(args[1]);

	if (channel < 0 || channel >= MAX_CHANNELS) {
		Notify(player, "Valid channels are between 0 and 5 inclusive.");
		return;
	}

	commSend(object, player, channel, args[2]);

	return;
}

void funCommFreq(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int channel;

	channel = atoi(args[1]);

	if (channel < 0 || channel >= MAX_CHANNELS) {
		Notify(player, "Valid channels are between 0 and 5 inclusive.");
		return;
	}

	FWriteFunctionBuffer("%d",object->shipdata->channel[channel].frequency);

	return;
}

void funCommLabel(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	int channel;

	channel = atoi(args[1]);

	if (channel < 0 || channel >= MAX_CHANNELS) {
		Notify(player, "Valid channels are between 0 and 5 inclusive.");
		return;
	}

	WriteFunctionBuffer(object->shipdata->channel[channel].label);

	return;
}

void cmdDamRepair(TAG *object, dbref player, char **args)
{
	int team;

	team = atoi(args[1]);

	/* valid team check */
	if (team < 0 || team >= object->shipdata->damcon_teams) {
		Notify(player, "Invalid team number.");
		return;
	}

	damRepair(object, player, team, args[2]);
	return;
}

void cmdDamSystemStatus(TAG *object, dbref player, char **args)
{
	damSystemStatus(object, player);
}

void cmdDamTeamStatus(TAG *object, dbref player, char **args)
{
	damTeamStatus(object, player);
}

void funContactListNumbers(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	snsListContacts(*bufc, object->contact_list, 0, 0);
	*bufc = buff + strlen(buff);
}

void funContactListDbrefs(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	snsListContacts(*bufc, object->contact_list, 0, 1);
	*bufc = buff + strlen(buff);
}

void funContactInfoLevel(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	CONTACT *contact;

	contact = snsFindContactByNumber(object, atoi(args[1]));

	if (contact != NULL)
		FWriteFunctionBuffer("%d", contact->info_level);
	else
		WriteFunctionBuffer("0");
}

void funTransListBeamable(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	snsListContacts(*bufc, object->contact_list, 2, 1);
	*bufc = buff + strlen(buff);
}

void funTransListDests(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	snsListContacts(*bufc, object->contact_list, 1, 1);
	*bufc = buff + strlen(buff);
}

void funTransPermission(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	TAG *target;

	if (Ship(object))
		if (object->shipdata->damage[SYSTEM_TRANSPORTER].status >= 
		  '7') {
			WriteFunctionBuffer("DAMAGED");
			return;
		}

	if ((target=objFindObject(parseDbref(args[1])))==NULL) {
		WriteFunctionBuffer("INVALID TARGET");
		return;
	}

	if (!CanBeamTo(target)) {
		WriteFunctionBuffer("INVALID TARGET");
		return;
	}

	if (!CanBeamFrom(target)) {
		WriteFunctionBuffer("INVALID SOURCE");
		return;
	}

	if (distance(object->pos, target->pos) > object->transporter_range) {
		WriteFunctionBuffer("OUT OF RANGE");
		return;
	}

	if (Ship(object)) {
		if (object->shipdata->shield_status[calcFacingShield(
		  target->pos, object)] == SHLD_UP) {
			WriteFunctionBuffer("SOURCE SHIELD");
			return;
		}
	}

	if (Ship(target)) {
		if (target->shipdata->shield_status[calcFacingShield(
		  object->pos, target)] == SHLD_UP) {
			WriteFunctionBuffer("TARGET SHIELD");
			return;
		}
	}

	WriteFunctionBuffer("OK");
	return;

}

void funRange(TAG *object, char *buff, char **bufc, dbref player, char **args)
{
	TAG *other;

	if ((other=objFindObject(parseDbref(args[1])))==NULL)
		WriteFunctionBuffer("#-1 OBJECT IN ARG 2 NOT ACTIVE");
	else
		FWriteFunctionBuffer("%d", distance(object->pos, other->pos));
}

void funOwnerNumber(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	FWriteFunctionBuffer("%d", object->shipdata->owner);
}

void funFacingShield(TAG *object, char *buff, char **bufc, dbref player, 
  char **args)
{
	TAG *other;

	if ((other=objFindObject(parseDbref(args[1])))==NULL)
		WriteFunctionBuffer("#-1 OBJECT IN ARG 2 NOT ACTIVE");
	else
		FWriteFunctionBuffer("%d", calcFacingShield(other->pos,object));
}
